<?php

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Components\Ajax\AjaxUtil;
use Avodel\WebDriver\Components\Frames\FramesHelper;
use Behat\Mink\Session;

final readonly class DebugHelper
{
    public function __construct(
        private string $dumpDir,
        private AjaxUtil $ajaxUtil,
        private FramesHelper $framesHelper,
    )
    {
    }

    public function dump(Session $session): array
    {
        if (!$session->isStarted()) {
            return [
                'isSessionStarted' => false,
            ];
        }

        $filename = sprintf('%s_%s', time(), uniqid('', true));
        $screenshotPath = $this->dumpDir . '/' . $filename . '.png';
        $pageContentPath = $this->dumpDir . '/' . $filename . '.html';
        $ajaxResponsesPath = $this->dumpDir . '/' . $filename . '_ajax_responses.json';

        file_put_contents($screenshotPath, $session->getScreenshot());
        file_put_contents($pageContentPath, $session->getPage()->getContent());

        $ajaxResponses = $this->ajaxUtil->getAllRawAjaxResponses($session);
        file_put_contents($ajaxResponsesPath, json_encode($ajaxResponses, JSON_THROW_ON_ERROR));

        $frames = $this->framesHelper->getFrames($session);
        $framesData = [];

        foreach ($frames as $frame) {
            $this->framesHelper->switchToIFrame($session, $frame->getPath());
            $frameIdKey = implode('_', $frame->getPath());

            $frameFilename = sprintf('%s_%s', time(), uniqid('', true));
            $frameScreenshotPath = $this->dumpDir . '/' . $frameFilename . '.png';
            $framePageContentPath = $this->dumpDir . '/' . $frameFilename . '.html';
            $frameAjaxResponsesPath = $this->dumpDir . '/' . $frameFilename . '_ajax_responses.json';

            file_put_contents($frameScreenshotPath, $session->getScreenshot());
            file_put_contents($framePageContentPath, $session->getPage()->getContent());

            $frameAjaxResponses = $this->ajaxUtil->getAllRawAjaxResponses($session);
            file_put_contents($frameAjaxResponsesPath, json_encode($frameAjaxResponses, JSON_THROW_ON_ERROR));

            $framesData[$frameIdKey] = [
                'pageContentPath' => $framePageContentPath,
                'screenshotPath' => $frameScreenshotPath,
                'ajaxResponsesPath' => $frameAjaxResponsesPath,
            ];
        }

        $this->framesHelper->switchToMainWindow($session);

        return [
            'screenshotPath' => $screenshotPath,
            'pageContentPath' => $pageContentPath,
            'ajaxResponsesPath' => $ajaxResponsesPath,
            'frames' => $framesData,
            'url' => $session->getCurrentUrl(),
        ];
    }
}
