<?php

namespace Avodel\WebDriver\Components\CaptchaVerifier\Verifier;

use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\Frames\Frame;
use Avodel\WebDriver\WebDriver;
use Exception;
use Webmozart\Assert\Assert;

final readonly class HCaptchaCaptchaVerifier implements CaptchaVerifierInterface
{
    public function __construct(
        private HCaptchaSolverInterface $hCaptchaSolver,
    )
    {
    }

    public function verify(WebDriver $webDriver): void
    {
        try {
            $this->doVerify($webDriver);
        } catch (Exception $e) {
            throw new CaptchaVerificationFailedException('HCaptcha verification failed.', previous: $e);
        } finally {
            $webDriver->getFramesHandler()->switchToMainWindow();
        }
    }

    private function doVerify(WebDriver $webDriver): void
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($webDriver);

        Assert::notNull($checkboxFrame, 'Checkbox frame was not found.');
        Assert::notNull($challengeFrame, 'Challenge frame was not found.');

        $siteKey = $this->parseQueryValue($challengeFrame->getSrc(), 'sitekey');
        Assert::notEmpty($siteKey, 'Site key was not found.');

        $userAgent = $webDriver->evaluateScript('window.navigator.userAgent');

        $solution = $this->hCaptchaSolver->getSolution(
            $webDriver->getCurrentUrl(),
            $siteKey,
            $userAgent
        );

        $webDriver->getFramesHandler()->switchToIFrame($challengeFrame->getPath());
        $webDriver->getAjaxHandler()->overrideAjaxResponse(
            'POST',
            'https://api.hcaptcha.com/getcaptcha/' . $siteKey,
            '{"pass": true,"generated_pass_UUID": "' . $solution . '","expiration": 120}'
        );

        $webDriver->getFramesHandler()->switchToIFrame($checkboxFrame->getPath());
        $webDriver->executeScript(file_get_contents(__DIR__ . '/../../../Resources/js/hcaptcha.js'));
        $webDriver->executeScript('document.getElementById(\'anchor\').click()');

        $checkVerificationStatusScript = <<<JS
window.hCaptchaSolved === true || window.hCaptchaSolved === undefined
JS;
        $passed = $webDriver->wait(5000, $checkVerificationStatusScript);
        Assert::true($passed, 'Checkbox was not checked.');
    }

    public function isVerificationRequired(WebDriver $webDriver): bool
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($webDriver);

        return $checkboxFrame && $challengeFrame;
    }

    /**
     * @return array<Frame|null>
     */
    private function getHCaptchaFrames(WebDriver $webDriver): array
    {
        $allFrames = $webDriver->getFramesHandler()->getFrames();

        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=checkbox')) {
                continue;
            }

            if ($this->isCaptchaSolved($webDriver, $frame)) {
                continue;
            }

            if (!$webDriver->getFramesHandler()->isFrameInteractable($frame->getPath())) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');
            Assert::notNull($currentCaptchaId, 'Captcha id was not found.');

            $challengeFrame = $this->getChallengeFrame($webDriver, $currentCaptchaId);

            return [$frame, $challengeFrame];
        }

        return [null, null];
    }

    private function getChallengeFrame(WebDriver $webDriver, string $captchaId): Frame
    {
        $allFrames = $webDriver->getFramesHandler()->getFrames();

        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=challenge')) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');

            if ($currentCaptchaId !== $captchaId) {
                continue;
            }

            return $frame;
        }

        throw new \Exception('Challenge frame was not found.');
    }

    private function parseQueryValue(string $src, string $key): ?string
    {
        $parsedUrl = parse_url($src);
        $query = $parsedUrl['fragment'] ?? '';
        parse_str($query, $params);

        return $params[$key] ?? null;
    }

    private function isCaptchaSolved(WebDriver $webDriver, Frame $frame): bool
    {
        $webDriver->getFramesHandler()->switchToIFrame($frame->getPath());
        $isChallengePassed = $webDriver->evaluateScript('window.hCaptchaSolved === true');
        $webDriver->getFramesHandler()->switchToMainWindow();

        return $isChallengePassed;
    }
}
