<?php

namespace Avodel\WebDriver\Components\Frames;

use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Facebook\WebDriver\Exception\NoSuchFrameException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;

final readonly class FramesHandler
{
    public function __construct(
        private MinkPhpWebDriver $driver,
    )
    {
    }

    /**
     * Recursively gets all frames on the page.
     *
     * @return Frame[]
     */
    public function getFrames(): array
    {
        $this->driver->switchToIFrame(null);

        try {
            $frames = $this->getFramesRecursive([]);
        } finally {
            $this->driver->switchToIFrame(null);
        }

        return $frames;
    }

    private function getFramesRecursive(array $path): array
    {
        $script = <<<JS
return Array.from(document.getElementsByTagName('iframe')).map((iframe, index) => {
    return { 
        index: index,
        id: iframe.id,
        name: iframe.name,
        src: iframe.src 
    };
})
JS;

        $framesArray = $this->driver->evaluateScript($script);
        $response = [];

        foreach ($framesArray as $frame) {
            $currentPath = array_merge($path, [$frame['index']]);
            $response[] = new Frame($currentPath, $frame['id'], $frame['name'], $frame['src']);

            try {
                $this->driver->switchToIFrame($frame['index']);
            } catch (NoSuchFrameException|StaleElementReferenceException) {
                continue;
            }

            $response = array_merge($response, $this->getFramesRecursive($currentPath));

            $this->doSwitchToIFrame($path);
        }

        return $response;
    }

    /**
     * @throws NoSuchFrameException|StaleElementReferenceException
     */
    private function doSwitchToIFrame(array $path): void
    {
        $this->driver->switchToIFrame(null);

        foreach ($path as $p) {
            $this->driver->switchToIFrame($p);
        }
    }

    /**
     * Switches to specific iFrame.
     *
     * @throws NoSuchFrameException|StaleElementReferenceException
     */
    public function switchToIFrame(array $path): void
    {
        try {
            $this->doSwitchToIFrame($path);
        } catch (\Exception $e) {
            $this->driver->switchToIFrame(null);

            throw $e;
        }
    }

    /**
     * Switches to specific iFrame and executes callback inside that frame.
     *
     * @throws NoSuchFrameException|StaleElementReferenceException
     */
    public function switchToIFrameAndBack( array $path, callable $callback): void
    {
        try {
            $this->doSwitchToIFrame($path);
            $callback();
        } finally {
            $this->driver->switchToIFrame(null);
        }
    }

    /**
     * Switches to main window.
     */
    public function switchToMainWindow(): void
    {
        $this->driver->switchToIFrame(null);
    }

    /**
     * Checks whether frame is interactable or not.
     *
     * Interactable frame means that:
     * - frame is visible
     * - frame is not overlapped by other elements
     *
     * @throws NoSuchFrameException|StaleElementReferenceException
     */
    public function isFrameInteractable(array $path): bool
    {
        $iFrameIndex = array_pop($path);

        if (count($path) > 0) {
            $this->switchToIFrame($path);
        }

        $js = <<<JS
(function(index) {
    const iframe = document.getElementsByTagName('iframe')[index]; 

    const rect = iframe.getBoundingClientRect();
    const windowHeight = (window.innerHeight || document.documentElement.clientHeight);
    const windowWidth = (window.innerWidth || document.documentElement.clientWidth);

    const isVisible = (
        rect.width > 0 &&
        rect.height > 0 &&
        rect.bottom >= 0 &&
        rect.right >= 0 &&
        rect.top <= windowHeight &&
        rect.left <= windowWidth
    );

    const style = window.getComputedStyle(iframe);
    const isCSSVisible = (
        style.display !== 'none' &&
        style.visibility !== 'hidden' &&
        style.pointerEvents !== 'none'
    );

    const isNotOverlapped = (function() {
        const midpointX = rect.left + rect.width / 2;
        const midpointY = rect.top + rect.height / 2;

        const elementAtMidpoint = document.elementFromPoint(midpointX, midpointY);

        return elementAtMidpoint === iframe || iframe.contains(elementAtMidpoint);
    })();

    return isVisible && isCSSVisible && isNotOverlapped;
})($iFrameIndex);
JS;

        try {
            return $this->driver->evaluateScript($js);
        } finally {
            $this->switchToMainWindow();
        }
    }
}
