<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Exception\NoSuchFrameException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;
use Facebook\WebDriver\WebDriverBy;
use Psr\Log\LoggerInterface;
use RuntimeException;

readonly class DebugDumpFilesPreparer
{
    public function __construct(
        private MinkPhpWebDriver $driver,
        private FramesHandler $framesHandler,
        private AjaxHandler $ajaxHandler,
        private LoggerInterface $logger
    ) {
    }

    /**
     * @throws \Exception
     */
    public function prepareDump(string $dir, bool $dumpAssets = false): void
    {
        $this->createDirIfNotExists($dir);

        $filename = sprintf('%s_%s', time(), uniqid('', true));
        $screenshotPath = $dir . '/' . $filename . '.png';
        $pageContentPath = $dir . '/' . $filename . '.html';
        $ajaxResponsesPath = $dir . '/' . $filename . '_ajax_responses.json';
        $pagePdfPath = $dir . '/' . $filename . '.pdf';

        file_put_contents($screenshotPath, $this->driver->getScreenshot());
        file_put_contents($pageContentPath, trim($this->driver->getContent()));

        $ajaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
        file_put_contents($ajaxResponsesPath, json_encode($ajaxResponses, JSON_THROW_ON_ERROR));

        $chromeDevToolsDriver = new ChromeDevToolsDriver($this->driver->getWebDriver());

        $encodedPdf = $chromeDevToolsDriver->execute('Page.printToPDF');
        file_put_contents($pagePdfPath, base64_decode($encodedPdf['data']));

        $frames = $this->framesHandler->getFrames();

        foreach ($frames as $frame) {
            try {
                $this->framesHandler->switchToIFrame($frame->getPath());
            } catch (NoSuchFrameException|StaleElementReferenceException $e) {
                continue;
            }

            $frameIdKey = implode('_', $frame->getPath());

            $frameSubDir = sprintf('%s/%s', $dir, 'frame_' . $frameIdKey);

            $this->createDirIfNotExists($frameSubDir);

            $frameFilename = sprintf('%s_%s', time(), uniqid('', true));
            $framePageContentPath = $frameSubDir . '/' . $frameFilename . '.html';
            $frameAjaxResponsesPath = $frameSubDir . '/' . $frameFilename . '_ajax_responses.json';

            file_put_contents($framePageContentPath, trim($this->driver->getContent()));

            $frameAjaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
            file_put_contents($frameAjaxResponsesPath, json_encode($frameAjaxResponses, JSON_THROW_ON_ERROR));
        }

        $this->framesHandler->switchToMainWindow();

        if ($dumpAssets) {
            $assetsDir = $dir . '/assets';
            $this->createDirIfNotExists($assetsDir);
            $this->prepareAssets($assetsDir);
        }
    }

    private function prepareAssets(string $dir): void
    {
        $pageSource = $this->driver->getWebDriver()->getPageSource();

        foreach (['link', 'script', 'img'] as $tag) {
            $elements = $this->driver->getWebDriver()->findElements(WebDriverBy::tagName($tag));

            foreach ($elements as $element) {
                $src = $element->getAttribute('href') ?: $element->getAttribute('src');
                if ($src) {
                    $path = $this->downloadAsset($src, $dir);
                    if ($path) {
                        $pageSource = str_replace($src, $path, $pageSource);
                    }
                }
            }
        }

        $htmlFile = $dir. '/saved_page.html';
        file_put_contents($htmlFile, $pageSource);
    }

    private function downloadAsset(string $url, string $folder): ?string
    {
        $currentUrl = $this->driver->getCurrentUrl();

        $parsedUrl = parse_url($currentUrl);
        $baseUrl = $parsedUrl['scheme'] . '://' . $parsedUrl['host'] . dirname($parsedUrl['path']);

        if (str_starts_with($url, '/')) {
            $url = $parsedUrl['scheme'] . '://' . $parsedUrl['host'] . $url;
        } elseif (str_starts_with($url, './') || str_starts_with($url, '../')) {
            $url = rtrim($baseUrl, '/') . '/' . ltrim($url, './');
        }

        $parsedResourceUrl = parse_url($url);
        if (isset($parsedResourceUrl['host']) && $parsedResourceUrl['host'] !== $parsedUrl['host']) {
            $path = '/' . $parsedResourceUrl['host'] . $parsedResourceUrl['path'];
        } else {
            $path = parse_url($url, PHP_URL_PATH);
        }

        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));
        $url = str_replace($path, $encodedPath, $url);

        $allowedExtensions = ['ico', 'jpg', 'jpeg', 'png', 'gif', 'css', 'js', 'woff', 'woff2', 'ttf', 'svg'];
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        if (!$extension || !in_array($extension, $allowedExtensions, true)) {
            return null;
        }

        if (filter_var($url, FILTER_VALIDATE_URL)) {
            try {
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $data = curl_exec($ch);
                curl_close($ch);

                $filePath = $folder . $path;
                $this->createDirIfNotExists(dirname($filePath));
                file_put_contents($filePath, $data);

                return '.' . $path;
            } catch (\Throwable $t) {
                $this->logger->warning('Unable to download asset.', ['url' => $url, 'exc' => $t]);
            }
        }

        return null;
    }

    private function createDirIfNotExists(string $dir): void
    {
        if (file_exists($dir)) {
            return;
        }

        if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException(sprintf('Directory "%s" was not created', $dir));
        }
    }
}