<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Exception\NoSuchFrameException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;
use RuntimeException;

readonly class DebugDumpFilesPreparer
{
    public function __construct(
        private MinkPhpWebDriver $driver,
        private FramesHandler $framesHandler,
        private AjaxHandler $ajaxHandler
    ) {
    }

    /**
     * @throws \Exception
     */
    public function prepareDump(string $dir): void
    {
        $this->createDirIfNotExists($dir);

        $filename = sprintf('%s_%s', time(), uniqid('', true));
        $screenshotPath = $dir . '/' . $filename . '.png';
        $pageContentPath = $dir . '/' . $filename . '.html';
        $ajaxResponsesPath = $dir . '/' . $filename . '_ajax_responses.json';
        $pagePdfPath = $dir . '/' . $filename . '.pdf';

        file_put_contents($screenshotPath, $this->driver->getScreenshot());
        file_put_contents($pageContentPath, trim($this->driver->getContent()));

        $ajaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
        file_put_contents($ajaxResponsesPath, json_encode($ajaxResponses, JSON_THROW_ON_ERROR));

        $chromeDevToolsDriver = new ChromeDevToolsDriver($this->driver->getWebDriver());

        $encodedPdf = $chromeDevToolsDriver->execute('Page.printToPDF');
        file_put_contents($pagePdfPath, base64_decode($encodedPdf['data']));

        $frames = $this->framesHandler->getFrames();

        foreach ($frames as $frame) {
            try {
                $this->framesHandler->switchToIFrame($frame->getPath());
            } catch (NoSuchFrameException|StaleElementReferenceException $e) {
                continue;
            }

            $frameIdKey = implode('_', $frame->getPath());

            $frameSubDir = sprintf('%s/%s', $dir, 'frame_' . $frameIdKey);

            $this->createDirIfNotExists($frameSubDir);

            $frameFilename = sprintf('%s_%s', time(), uniqid('', true));
            $framePageContentPath = $frameSubDir . '/' . $frameFilename . '.html';
            $frameAjaxResponsesPath = $frameSubDir . '/' . $frameFilename . '_ajax_responses.json';

            file_put_contents($framePageContentPath, trim($this->driver->getContent()));

            $frameAjaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
            file_put_contents($frameAjaxResponsesPath, json_encode($frameAjaxResponses, JSON_THROW_ON_ERROR));
        }

        $this->framesHandler->switchToMainWindow();
    }

    private function createDirIfNotExists(string $dir): void
    {
        if (file_exists($dir)) {
            return;
        }

        if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException(sprintf('Directory "%s" was not created', $dir));
        }
    }
}