<?php

namespace Avodel\WebDriver\Components\CaptchaVerifier\Verifier;

use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CheckboxWasNotCheckedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\RecaptchaV2SolverInterface;
use Avodel\WebDriver\Components\Frames\Frame;
use Avodel\WebDriver\Components\Frames\SwitchToFrameException;
use Avodel\WebDriver\WebDriver;
use Exception;
use Facebook\WebDriver\WebDriverBy;
use Webmozart\Assert\Assert;

final readonly class RecaptchaV2CaptchaVerifier implements CaptchaVerifierInterface
{
    public function __construct(
        private RecaptchaV2SolverInterface $recaptchaV2Solver,
    )
    {
    }

    /**
     * @throws CaptchaVerificationFailedException
     */
    public function verify(WebDriver $webDriver, array $frames): void
    {
        try {
            $this->doVerify($webDriver, $frames);
        } catch (Exception $e) {
            throw new CaptchaVerificationFailedException('RecaptchaV2 verification failed.' . $e->getMessage(), previous: $e);
        } finally {
            $webDriver->getFramesHandler()->switchToMainWindow();
        }
    }

    /**
     * @param array<Frame> $frames
     *
     * @throws CheckboxWasNotCheckedException
     * @throws SwitchToFrameException
     */
    public function doVerify(WebDriver $webDriver, array $frames): void
    {
        [$bFrame, $anchorFrame] = $this->getRecaptchaFrames($frames);

        Assert::notNull($anchorFrame, 'Anchor frame was not found.');
        Assert::notNull($bFrame, 'B frame was not found.');

        $this->scrollToFrameIfRequired($webDriver, $anchorFrame->getName());

        $parsedUrl = parse_url($bFrame->getSrc());
        $query = $parsedUrl['query'];
        parse_str($query, $params);

        $siteKey = $params['k'] ?? null;
        Assert::notEmpty($siteKey, 'Site key was not found.');

        $solution = $this->recaptchaV2Solver->getSolution($webDriver->getCurrentUrl(), $siteKey);

        $reloadResponse = <<<EOF
)]}'
["rresp","03AFcWeA7djpBXxOtBY3v7isnFUdQRlBm9z2HPKQWrTMG6Thkoa1Z0FzwYGHW0rLKu2etzMz4sOaUaNt5VL9DzyzVZQt6CglOUy5Q62tT1fIsXv3OLh1n-R4e3w-oeN0d6j9ncCc_TSwpNuOS6lhoJ2wjb_NAQAUZ_Z0fF1vNQIq02zJdGPOb2lMx4JA0KC1MG_xZgsar3oL7fsKV4hy8qy9va1MYIuELH0gA9xQ1ROwjZETubF1QrFwvoCF-6OjeitjVXEVf5UJJV3moIF_zmM2hLQkWh0aqHMZcQB27AbenOqTWzkq0ah2ifWJ-JzasiWY7yO8fiYfQh3AVRUtyTcBAFfHZVUaIzRkIg7JkcAIU0dMveNvNPXyMedHKjIpEklWG7BFuN1nL1VSufRmhM-bKfzsIOaVLEwa6VoEIPxDj5EFev9Re3PjkZuqqL3twavRu809jvQJeYznXe--vNogso99BzeeCAby47jxInrVqlXvRf2bdpR5yOF87Ems2t5uI6ucK9mhbI6jOPb7VuU5Rgzon6iOSTjvGx1kUTpxGemlDz38_mQdmjsHrxekx5BnGD6wpHKKy1CYRGR2FaoQFamVQWRGbUmm1AKDtpU0EapDsbp38wrJLYwCDPrLtnRf1WAIt6uSvOCM0tHq6gKIKk5s1Ts8UNkfPbQEXrdJ_7D7hT0r6ZJ5jKSAmFc9k1e7XFslfWqOpgH46jzZoJ17WzoEEnj0r48L770CdNLkc-615dMZ2N9JQ1UNSW0lGPygKQaYlbmd134D9m4qgFoBwwqHl0MPxdKb6V5TX8M1AieRuAJU71p7yEbe0uNVEt43AvwabIawHWkevLRnAo55O2CutZS-PJ4EIJ_6MANZ6LC2fAd27PammyIdDqtx7znSM8n2bDf4OqJb10yXxb816x4nrBMLQHyJm7gdgOZnOb2SZ4-klR0oejluVbp56cisthW9OGp1RwFSoVmQa6kK1VcFLOjArtUiOAcmTBL_2-CQFZqyTQMWM5QEoWNnz3MiOYN78bYPjS8yXiTa_Ggwo4-D_0z-zMitlTtImsz_9p3ovpjQlcniHIHnYIH7L5B4ImkCMVtZbIZ2nanz7ikFPgQ_JJOuL-GVjmdk506hJx821Vun-e69azYxe5XBzXx3oDPeZFqmP2amrLnOeI7ga3M3ILBDyKTNeIx2vW5Ht9tfmDpkH3xeYtUtWkmtewYtOoVypItrHBGd_Klx5rps42Nvx2w5DPV6pJXzBs4J-iampu-HPBaezNo7D7UNexYL2gyxYDCQ9L2I5E8GP6d6Yo25DyfDnJ1-H93NFCyiJ6obvhtEQtF2Ts3b25RNRJGBIGatc2HirVuzd8dhVBYrr-0P7Q9SUTC2dZWnz3sQXZQUWXnTIw4Fwm8zEO_v7czfGSNpiU5wXC0kPYYI_v4JVLOVGWh4Ae95fNmSf5ZBAgpDIs28yLndfJS1kFQZCG9akGDKIX_JjhDBfZLK0JGbvjGIZCgwVVcO_WULTYUtiS8etf4LbuE0xlW-3MkRRgkY-p1NPlVlv2LrbvArTmqXbQaIzAubqozmHyzj6hivShujgENaa-5CPH3ARSJbgsILlmrUjxq2dWX14D9z_1FT0tTGkZu9Ogiuyd7C0PsNktVAQFp480c_DDPvutuOmsj-kJDy_Ezpos8K-zSUOGpzfEc9jrw_mUydtIku0gFRHGRn8BZDO5j-9jmQ0zAC_9lWa61hhk1b3I7tlB45jFxXPNFXQagC2mbPViyaIelxA5sCasRDIy4uCxWUhW5FUSg5rfYU5fyLu33qnRgrcbkBRTEge-YdhtQEZwJ0a9vl4El6edo7gib5ZuGtCEXrcWB54D9LVXTWNXEPm-i3e7FXS_cjCAQ-QVK6Fim5qGybrShMqUVllaEZbzDkhjzOFE5f5_jv2ze-0iZiCD6tFXDIqiGeKcQR_A_4Twvy7ssWkqwmD6LaWq4RlDA6AnarRCPD6_23eE-iSZSHl3uc7Xo_2CBJ3ZSVteUq4vkTXZJ9rNRWGNemBWwvi-YtvLUw_IH8zbRkaIJt2KtbEu7cMITyZVsblmOgAmN-GdqftHiQvjj0klcK9tFy3KlHQw_Gk8zzo0A-ZtHzducgS-WdQBjDHdgkc1ipGAQ29jKkFoeO1OFls-MVOhyN4DSs1NVlBsfDi5Oid_PPQhP-6WgJYqkdInH4vc3qut27nU8fcYh6tNRDGTHKBh1I39dVnKsMiE0XMlma1gPVF6s4H5nvQKa8YFw3ZFs_Q7GnTQYKHppwcuOkz7J7kudKUFPxYXgO4q6EIbS4R8RlcpZGWDEWpgfP36yNuIfDOHuxtAKBMNhN8XTN5y6P0B3kou8KmApboFRbVbJ8E025j9DhMC4JaIjKsCVotLmpNSZdnvhFnUjZ3MkngMRRPhiUfQk66jO0iQvTvMfFfqYqThcOYT9TAdlGhn8uu9sIzHvN-fsgPI_ExQhUb3P3XEK3YjqVJUlxlJOZj7z2xbaENXivKH2OL-MR5qr1B0tVBwsBv7c7KOJ_ISF6qmem66pNrVcybiOrKH7xzrIk-RNy4PvScp-CtSB6aFy1NnG1BEgo-IplTeknBZzQi5Zi2OGCiKK9pF8LgQ-SHCuKtGA0HvC1Haf0OjQHnT1WIvQZVg6Zgs9ibEmVvSut73Vmr6fd8_BOfRj8yB6Er_SvhPVvxKM-sAniSyqDJJCr7S9OQkvtsZqflEINDPQFI5odzhK4k6ykI6RbgGymR-aUXFGN7dwuLHLm5FFFqAh8NVji8SV07gU_z4lxHjFRo4nPdYtgeIu88qzcL21Hp2o0oFJBj5oHxigxFx-kWM6faCuXTbFgoxJhCztSuKBlzmGCkCVToYzX0gJPEW1FWXvcrVrKk18jhKW3MmR60oUFhr__szbASRAN1ZwUdB6rEvKA3O_9PcYzOd8Ya5tVZSHuEkjlvYMWVsXZ_3K2OPyH5CKnwqg-sF6wHtDqPpwnlVdTqCgKULmSiWIaQl9XNfvzfbzNyMUHpNQz_c97Mfvna3toOA4GtP_ddf4dyakBW088Z7RJ-rrwh5Eyp6pI4-KhDAaMWqDY0BpDH0pK5vDJ6BBDUwMXvQRTHliZpH6TU1ctLod5EjxVMJeMTmK9ZtzeKoguTjk0k-I3qAkx0vd91CWGGF3Ip8hdTWSRSOFjy14Oi9cDFkM00_OI4z25q51xRjVItiYYynAqAWy27o89Hh9CD251H-f9HycIO-r49YFzrfsZc0gBZX5cwSJWKyx8qpJ1wDzMp1W8IMU6OTwQL9Mk0CcU0uwrX4KGq5BHc6ZfCd3sTkMIuAQiWvUjdVGXMfsZi6s7jTE4uTwiv9zfQgvhvb-hAAWbxI1t_n4hWZ2orCeDFRZBiT_uY-JwW5C4kn1X7zwEjG1NUquo8Mwv8w4lvjPUyA8iRv9PtrMvPlmgGUrW2eukyVshqiKBZOWuIAncaz2ksSUm2amHYtCTySmydiHhwr_BJ5euaGevP8brjVaizrMY71B5nGr3LJhtzDuB3wqUYhFlZD5Qk1eooeK0vXFU1BKXttMy_otau55t3IcCrRl_M3m-39gfYbY_pak9KVRmx2YLwREO1Mn9BOgYTgeKDBy2xN-Cnz7cMnNm7rdSCMf07CvDzpF4PM6zFP68podl2UQsK69kzF4WnojLqE4LEtUMipFPlAZxO6ny3KbZ4Vwjf99uPl9-JtF8eyyfM2v3jPV77WmAyCazLngi71K0OsLXd73-LBItzxcrjaWIKmnoA1CUs5gxO7zIjV1rDymVgAQ4jnglehNCbQ5utGxfs8WKmGGWNFOyxkktkxBUHbtWPsyPAX5lfSU_oMXb-MYMXfzooFYuYp5s7pc7WTbUO_UlfNIAAHoe3NgVeskKy8rG8uhn6EnMWcPtMErzu9r6N8IuJzxzyqfCcWoYSMCb-f6rYV3xUpmV-Z1qiYujos4vCk_wNV2yR3V2lj_38BhsFaHRW6FnJMOutwmAEvy5BiCgvT2nFucAfOlNBjLfu0eQ4Gu8khcGaB78hBKXF7GAI4lnPo6TfPHKH3r0hqlxIYYy8Ca4WrJFwfzUhxeReEziHvk1DkQfXNua5HuoJnxLiGLRC1BU5hjiGLCulDmt6QfD_CyFZVifAClO6gAxWz1zX3dB8mtTZ8cOTla5yO3sVRiKn-wuupoFlDzGlAMkq_rUGbZs4GcIdjvFqNVGMpQMsJDgxAYdkkJoxWRKJgSRNM9qFiNw1-17zSZVCs-6--Y7APoxbk-NYstiUvl3baynlqpjbWzuHP1_oLS2FE-qKFqVUhiY--t8u59GnopjuZqUKGFDYOy1bwi8E3fpDqb7Hc6qf8LHPf5AbgSnH25I_PRiEsTU85B37rsBHGBDJFPRoysc7_4T65Ks6TJl3zglEAPD_KWc820Rgpzb5WS-ULO-3Fw7GOF7GjclZjMJpDdm8kUGGEVGmREpUEkkILdfWt1IdYclbMrukdZ5bkOnr1r1YENmo_zhEZvP2G4jqpi_Fe2yY1ZWL_m7MZZ7YXqMQ01A7D6wavyG_-nNGSr81l_NNaG-hz8ZAuTO88oEBHaM1TNBvYyY2ATXTVAc-Ixm8sg8BqugPvnxFpc71UMgB5q80uunNuOYGVmuX1ounHgrWFoY7tmu8OFNCXph6UbK-oMiQ09ymFS9CKGGNB5VFiT32cZ5qxeHebUfHppjySZo15qL5L5AFKxsbPtZioRL7xlSQPl2UhB6u7GyI53CSpOl_j7LxUN3hvI3URGk4XXbl9n52wxKEkof_0mqUIuZV_oIvWdUckTivUTcqtZo0OO1SSa7flpj-ATvy6sQQ_P_kZpDlYV4lZv558U3PdlVe9GaGAWT-mio8HkJ79b7_BmT56VqbSfnLPoZI7ZkFjmbp6pqq6IfnTh_2qyo0cjV5ctI8z8VL49bS3S5TJFEPoBWasB_AmN6HxyRojxTUC0qTVE4HhYD1rslVS3PnURF16GuOlsP5irvgaJYiUW0IHnjXq_idzDP8DoeUjVCoRgnqa4BcJh5l0V67b7ls5i5It72t68w0OMHppR4vmUwqqLjwuVKgtMfq3NA0ymGbmW4ugb3pVcArTrAGCrfbPzSyEDk3Om6H2d7vutKVsEXfnH0rO5Jvz7GHcK1enZ5WyjhkxPxs-2oBugiWZgho2WeZxx3z8Pd3J4dls-tahBtdslZX_iZ-EgPbbW6Hr3FY3YOkP3YN4fJIGoXAxarAJ8enD7_iz5SHmJ0O-71enJa1DNqbB5rVmNXLxKxQAH4WChNonS4_nTvlLl2eUX39QzD-N3Jn9KZ-s0OmFtjan_DPnPyPbrz03VMmD6-Tm6YNsPY9RLlaSiAzoU6fJQx3g4olzMZyYQO-3sknOcPwcFj-gq42rKLAcpI4bbCVu66u3kvU7CEJUyiuA4yFlD-7gFjTBDdsFv1uWCpcHnBEeuNN3U_-aWf6QNQ6VLSzC5T-WFnvzXCktkoKsCruvN44n8G8Amc18VpqvdGBkK8OsCdQhluU62aNzLX4EXZKaTCCREigsfWXUsN4YBvpBioPX3CGo02wV6H06Z_YXszncwBxK0p9WObUw1W-jULr_6ap-cxSbx9Ctv8Ku3blf3_dRCcUFlON0JmmSJx9QEdwOE69HSrZ9gfLTqjFdxhPHIeH4iEr_Bqsu4veh34ug5oOmdqWeABQE11-_So7O1Q2DiSoJrNgc5c7SGaJDDYAq7zNkA4K65x_Ff-i270gHVNoazIzkITWwSn_MzVZ92FarNznjnql5uiNcbU8PlOdIF2LLOW-JCZISFmKVbGlZJOjBR9zUuw8huc6kVcrg481YLhm4QSKeTqp7ejxK_RxAwhd9q9KNgvvja3ZuT0GsW6Va-zVoEl7pN4FxxoTOI6C2Qi9-vKm7x_gAibOb1nwxVALUU_22X-66a05hsMFzwRui-XjmbFXHs1NvGbUV_eOoIf9RNTSnNbxHqxK-GhJixpbpYW96Tqo5_aaw9n8ssmQ3Xz7mWMO4CLlJ2Mq8a5i6zoO6xUfrw5yniC1gGfCbh19Fn518UdvBl_nZP0dkF_22y2VvZ06Xa4clKYmviOjhsYpXhDKDfBV_RAoyz7x-_kEzKePsrQkx_9ItR-r67QKrqeDgd9VhO1xSATv3UWg33AjDfguD9bqRcy4Y4yCN3_EZCTIZKMquXf35VFf-4F_yb2WNNNiEmuUAooK3tmxHCDtImyhw3qTtuxImIlaIGFXQ1MA4NJ7QMFuTI3J2A7a09wTZaWJCeWrSh8s9mDP5h6-NgrOTdyKze-2jQy9xWPY--xLn-1ho5cItMR2MPsVuo-E-55NuE_u7IQM4ThqkqWpob7HarAN37K_6nTX5_b16EQKlHSqYhKbjK64jpQ5XWcr9jsRyuC-h4q7FDSwMmcf2sNTgFrk3bPSj1r-BZ88ik0PXG6-uE37Stku90qsNJmX7LOHTHwBDp3xKOcQEHkqqslDZ1kOsDGxB9qKHfQYGgU7pxSQk2y0WNzSBM0Pvauc08GrAAepFFtRYe9CK5BBYUx19zt-zdM9hLDa1xw",null,120,null,"nocaptcha",null,["bgdata","Ly93d3cuZ29vZ2xlLmNvbS9qcy9iZy9NMFBvVUVKbF9mbkpZbWdvWThGRC1mSkRySzBsUnZnTU5mVUF3UFdvX19NLmpz","","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\u003d\u003d"],"05AEK4rYF2QtfR-2kJR18TNpO8iQprLHng9Wic7gZbMVfW2bAxLnqrQFb7IYLg3tbwN3Ji2iE36ZfOGZ4iDlAfXGUYN2O5DgnhnB_4EC6uAORMGP55rWK0eQmoWDbR_kyYdVZakXl7joHfHFmH45PUmYaVGMCuY36afOtlO77m3KjA3cntc5e88bCeX6gr2WpuR2OV6F6_Zjcz877RPeaB5JT8HtvAMzpxARotfrxyTLbTKdcWvz88mlYwUrn0lXkd-2scNTyz1tt62bekL_WC74gOnBpdR5DmGS4C6FujE2TcnPsqzatisK7dPnOUOs58w9gjWyOg_gbqySvSmiHwc0gG3R3YGNef2VDdo0UPW1A3EfOWk6_W0zGYpFZSHsFcctC08f5_YsHdi6PihMO6xVjuNZuCCQf8K8C1FtJ4r-rewwgZSN7ZcNysLoAAkLw4YiUDUQxeg9XmN9JWW86seJBUUmsUwJ3uq-4Q9RPAkqSCgV5wYwv-EnvjoAHXeJ29ooEECUp1eXAg8VLDBKwsDbzX06UuWhu6CBGWsj1UVcZ7f-Su-9IN9dcFXPly27Dl0R1ka8vJnXvdyrZhCXMbzkVzjheN4kCQAI9F4fcRtpj4edE-N1yeAR3ByigcP0b_0gblSfT4UZwP7xUpPhdTGkkCUWtKK5SIvxSK-s8TDYqwMKwn44anKxqrT1SZ9gVaEH44r8KX53AikLGBiNZGoultOlHYjNZWtKw9BYmb2G-Q_ptICVO6MBVM6ZhtymN5bJr4HRZwVMPRGWlFcJ98vnk4AigkQQlTAgg6zxomxzVuorj3qwSy0Pz4yUMkeow4k1P7iO1CUNSs7MKaCMBVSw-TE42-2REeO7g5bartpv5BFg-QoxvPpkXd7kW8TCQz0o7kCvO62YdSr4rKi-Lo93f2_ni5euvrnV8WxAmaRE42Eimlr8C2m0bzt5UXgkm_n0WCgphepefHBFsJQqk45PBpXVCPeajpn7NWOmUVso_k7na5kPKMfFPsnQTZWyxSi91giK1SSfpaUCBBYgRitbVyEMdyV7-VonRDCt6qulbOB1jnJZ99oEHtEiH5sOpj2rjtqYeqr8b53Z-b0Db7v8YPEF1b-7h8D-FGlNV9tUvQXn-2O6ID5FNXRasPcwwe00hg9RU4rXM_Tgn21-y5bKpVjKDNcNvCQf_MNH_zbtDfjpWOencgoQ454MHMLp6VWX4C5_FMmbYIE6EUavAMQ7xPEVHYY4DYKvAoxkT4yOfa3suWd7d3TaBYgYmg8u3V_VmHTOw-p_zuDqXMtQREhBKxgrGlV62vbtWCbPFG6ApZ3nfxj6qUeVrQajiF1tm9XJdMBHMKjZHGUpKv_k5pJxClBGYdPq2uAWPlM60Aqto","06AFcWeA6CPQ2q-NI_migz0AJ385EbMU9yZLCQlA5bwxv-seT8p5nHpN6rR3eqKo5wzZfdwN5oyIgqTM5zJMosQjokFc6bmvjHXED-z7Xy2yXanBHJc0c6yiIlFWGUVvcOxXLFVJ1pAwzZAiV-3ZKDEj7vxbr2syWwQBVmSnuQa037OAqii74ekxIH-hT9Aeg4KAEjKobg5kgW9C1QX36oaZpjlS2Rd7qpYXQdD1mswGIbzURzJgOE5WM",null,null,"09AEK4rYESy0bIPz5OXDANw-Ui7mI3NtGNwnfEuDnX_I1hcAaz-8ewI_RALiQhLHYJ9hWk6VCcpYIGVKpQ3_XSr0YMVcidQiQgRKeesQ",0,"0aAEK4rYGNrfEW56l7oHY5LLIwHpTbJltt7xR7ZlcUup3D0dXIj_J6nf3jW7f9vK3CAAkKIByZOIWFMfh-BGQco4Ua7tyuWL7_plW1GZ2AjUSYtB53wA","0bAM7a3JEgFbm30Ioxt8hbuie_4K340iwuwF6OqSCfH1c4No0K3CnKDi-ww7_exUwoc8v1cyyWPA",0]
EOF;


        $userVerifyToken = <<<EOF
)]}'
["uvresp","$solution",1,120,null,null,null,null,null,"09AEK4rYFBWcModM-Lc7i15vDL8hJCtE1y8p5FVU3EshhkKv8mb-JTO8_aTBlvBYgKucpE152M_kZpSmX-SsteGipiC1WwnQ_ZY_y8Xg"]
EOF;

        $webDriver->getFramesHandler()->switchToIFrame($bFrame->getPath());

        $webDriver->getAjaxHandler()->overrideAjaxResponse(
            'POST',
            '/recaptcha/api2/reload?k=' . $siteKey,
            $reloadResponse
        );

        $webDriver->getAjaxHandler()->overrideAjaxResponse(
            'POST',
            '/recaptcha/api2/userverify?k=' . $siteKey,
            $userVerifyToken
        );
        $webDriver->getFramesHandler()->switchToIFrame($anchorFrame->getPath());
        $checkbox = $webDriver->getPage()->find('css', '#rc-anchor-container');
        $checkbox->click();

        $checkMarkerScript = <<<JS
document.querySelectorAll('.recaptcha-checkbox-checked').length > 0 || document.querySelectorAll('.recaptcha-checkbox').length === 0
JS;
        $checked = $webDriver->wait(1000, $checkMarkerScript);

        if (!$checked) {
            $checkbox->click();
            $checked = $webDriver->wait(1000, $checkMarkerScript);
        }

        if (!$checked) {
            throw new CheckboxWasNotCheckedException();
        }
    }

    private function isCheckboxChecked(WebDriver $webDriver): bool
    {
        $script = <<<JS
document.querySelectorAll('.recaptcha-checkbox-checked').length > 0
JS;
        return $webDriver->evaluateScript($script);
    }

    private function hasCheckbox(WebDriver $webDriver): bool
    {
        $script = <<<JS
document.querySelector('.recaptcha-checkbox') !== null
JS;
        return $webDriver->evaluateScript($script);
    }

    public function isVerificationRequired(WebDriver $webDriver, array $frames): bool
    {
        [$bFrame, $anchorFrame] = $this->getRecaptchaFrames($frames);

        if (!$bFrame || !$anchorFrame) {
            return false;
        }

        try {
            $webDriver->getFramesHandler()->switchToIFrame($anchorFrame->getPath());

            return $this->hasCheckbox($webDriver) && !$this->isCheckboxChecked($webDriver);
        } finally {
            $webDriver->getFramesHandler()->switchToMainWindow();
        }
    }

    /**
     * @param array<Frame> $allFrames
     * @return array<Frame|null>
     */
    private function getRecaptchaFrames(array $allFrames): array
    {
        $bFrame = null;
        $anchorFrame = null;

        foreach ($allFrames as $frame) {
            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/anchor')) {
                $anchorFrame = $frame;
            }

            if (str_starts_with($frame->getSrc(), 'https://www.google.com/recaptcha/api2/bframe')) {
                $bFrame = $frame;
            }
        }

        return [$bFrame, $anchorFrame];
    }

    private function scrollToFrameIfRequired(WebDriver $webDriver, string $frameName): void
    {
        $frame = $webDriver->getPage()->find('css', 'iframe[name="' . $frameName . '"]');

        if ($frame === null) {
            return;
        }

        $element = $webDriver->getRemoteWebDriver()->findElement(WebDriverBy::xpath($frame->getXpath()));
        $webDriver->getDriver()->scrollElementIntoViewIfRequired($element);
    }
}
