<?php

namespace Avodel\WebDriver;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\CaptchaVerifier;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\CaptchaVerifierInterface;
use Avodel\WebDriver\Components\Debug\Debugger;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Driver\DevToolsDriver;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Behat\Mink\Element\DocumentElement;
use Behat\Mink\Session;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use Facebook\WebDriver\Remote\WebDriverCapabilityType;
use Psr\Log\LoggerInterface;
use Symfony\Component\Serializer\SerializerInterface;

final readonly class WebDriver
{
    private Session $session;

    private AjaxHandler $ajaxHandler;

    private CaptchaVerifier $captchaVerifier;

    private FramesHandler $framesHandler;

    private Debugger $debugger;

    /**
     * @param array<CaptchaVerifierInterface> $captchaVerifiers
     */
    public function __construct(
        private MinkPhpWebDriver $driver,
        private LoggerInterface $logger,
        private SerializerInterface $serializer,
        int $maxAjaxWaitingTimeMs,
        array $captchaVerifiers,
        string $debuggerDumpDir,
    )
    {
        $this->session = new Session($driver);
        $this->ajaxHandler = new AjaxHandler($this->driver, $maxAjaxWaitingTimeMs, $this->logger, $this->serializer);
        $this->captchaVerifier = new CaptchaVerifier($this, $captchaVerifiers);
        $this->framesHandler = new FramesHandler($this->driver);
        $this->debugger = new Debugger($debuggerDumpDir, $this->driver, $this->ajaxHandler, $this->framesHandler);
    }

    public function __destruct()
    {
        if ($this->driver->isStarted()) {
            $this->driver->stop();
        }
    }

    public function start(): void
    {
        $this->driver->start();
    }

    public function isStarted(): bool
    {
        return $this->driver->isStarted();
    }

    public function stop(): void
    {
        $this->driver->stop();
    }

    public function visit(string $url): void
    {
        $this->driver->visit($url);
    }

    public function reload(): void
    {
        $this->driver->reload();
    }

    public function getPage(): DocumentElement
    {
        return $this->session->getPage();
    }

    public function wait(int $time, string $condition): bool
    {
        return $this->driver->wait($time, $condition);
    }

    public function getHttpProxy(): ?string
    {
        $httpProxy = $this->driver->getDesiredCapabilities()?->getCapability(WebDriverCapabilityType::PROXY)['httpProxy'] ?? null;

        if (!$httpProxy) {
            return null;
        }

        return 'http://' . $httpProxy;
    }

    public function getDevToolsDriver(): DevToolsDriver
    {
        return new DevToolsDriver(new ChromeDevToolsDriver($this->getRemoteWebDriver()));
    }

    public function getAjaxHandler(): AjaxHandler
    {
        return $this->ajaxHandler;
    }

    public function getDebugger(): Debugger
    {
        return $this->debugger;
    }

    public function getFramesHandler(): FramesHandler
    {
        return $this->framesHandler;
    }

    public function getCaptchaVerifier(): CaptchaVerifier
    {
        return $this->captchaVerifier;
    }

    public function executeScript(string $script): void
    {
        $this->driver->executeScript($script);
    }

    public function evaluateScript(string $script): mixed
    {
        return $this->driver->evaluateScript($script);
    }

    public function getCurrentUrl(): string
    {
        return $this->driver->getCurrentUrl();
    }

    public function getRemoteWebDriver(): RemoteWebDriver
    {
        return $this->driver->getWebDriver();
    }
}
