# Web Driver
* Useful chrome extensions are available out of the box
* Captcha verifiers are available out of the box
* Uses the latest Chrome version
* Uses mouse over and keyboard actions to interact with the browser
* Uses Mink interface to interact with the browser
* Provides a flexible adapter over the main remote web driver.

# Configuration
```yaml
version: '3'

services:
  test-runner:
    image: registry.gitlab.com/mypie/docker-images/frankenphp8.3-test-runner
    volumes:
      - ./:/app

  browser:
    image: selenium/standalone-chromium:latest
    shm_size: 2gb
```

```yaml
avodel_web_driver:
    host: http://browser:4444/wd/hub
    user_agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/127.0.0.0 Safari/537.36
    components:
        debug: ~
        ajax: ~
        captcha_verifier:
            solvers:
                cloudflare_turnstile: App\Context\Login\Captcha\CloudflareTurnstileTokenSolver
```

## Using
```php

use Avodel\WebDriver\WebDriver;

class Service
{
    public function __construct(
        private readonly WebDriver $webDriver,
    ) {
    }

    public function execute(): void
    {
        $this->webDriver->visit('https://www.vfsglobal.com/');
        $this->webDriver->getDebugger()->dump();
        $allAjaxResponses = $this->webDriver->getAjaxHandler()->getAllAjaxResponses();
        $frames = $this->webDriver->getFramesHandler()->getFrames();
        $this->webDriver->getCaptchaVerifier()->verify();
    }
}
```

# Extensions
Extensions enable the configuration of the WebDriver by applying additional settings, such as:
* Timezone override
* User agent override
* Navigator platform override
* Navigator WebDriver flag override
* Ajax overrider component
* Cloudflare Turnstile solver
* HCaptcha solver
* Recaptcha V2 solver
* Navigator languages
* Hairline fix

# Components
## Ajax
The Ajax component is a tool for handling asynchronous requests and responses in the Avodel WebDriver. By using the AjaxHandler class, you can fetch, override, and wait for Ajax responses.

Usage:
```php
$ajaxHandler = $this->webDriver->getAjaxHandler();
$ajaxHandler->overrideAjaxResponse('POST', 'https://google.com', '{"key": "value"}');
// ..
```

## Frames
The Frames component is a tool for handling frames and iframes within a web page in the Avodel WebDriver. By using the FramesHandler class, you can get, switch to, and check the interactability of frames.

Usage:
```php
$framesHandler = $this->webDriver->getFramesHandler();
$frames = $framesHandler->getFrames();
// ..
```

## CaptchaVerifier
The CaptchaVerifier component is a tool for verifying captchas on web pages in the Avodel WebDriver. By using the CaptchaVerifier class, you can verify captchas and check if verification is required.

Usage:
```php
$captchaVerifier = $this->webDriver->getCaptchaVerifier();
$captchaVerifier = $framesHandler->verify();
// ..
```

## Debug
By using the Debugger class, you can dump the pages state including the content, frames, and ajax responses and screenshots to a directory.

Usage:
```php
$debugger = $this->webDriver->getDebugger();
$debugger = $debugger->dump();
// ..
```
