<?php

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Exception\NoSuchFrameException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;
use RuntimeException;
use ZipArchive;

final readonly class Debugger
{
    public function __construct(
        private string $dumpDir,
        private MinkPhpWebDriver $driver,
        private AjaxHandler $ajaxHandler,
        private FramesHandler $framesHandler,
    )
    {
    }

    /**
     * Dumps the page state including the content, frames, ajax responses and screenshots to a directory.
     */
    public function dump(): array
    {
        if (!$this->driver->isStarted()) {
            return [
                'isSessionStarted' => false,
            ];
        }

        $this->createDirIfNotExists($this->dumpDir);

        $subDir = sprintf('%s/%s', $this->dumpDir, uniqid('dump_', true));

        $this->createDirIfNotExists($subDir);

        $filename = sprintf('%s_%s', time(), uniqid('', true));
        $screenshotPath = $subDir . '/' . $filename . '.png';
        $pageContentPath = $subDir . '/' . $filename . '.html';
        $ajaxResponsesPath = $subDir . '/' . $filename . '_ajax_responses.json';
        $pagePdfPath = $subDir . '/' . $filename . '.pdf';

        file_put_contents($screenshotPath, $this->driver->getScreenshot());
        file_put_contents($pageContentPath, trim($this->driver->getContent()));

        $ajaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
        file_put_contents($ajaxResponsesPath, json_encode($ajaxResponses, JSON_THROW_ON_ERROR));

        $chromeDevToolsDriver = new ChromeDevToolsDriver($this->driver->getWebDriver());

        $encodedPdf = $chromeDevToolsDriver->execute('Page.printToPDF');
        file_put_contents($pagePdfPath, base64_decode($encodedPdf['data']));

        $frames = $this->framesHandler->getFrames();

        foreach ($frames as $frame) {
            try {
                $this->framesHandler->switchToIFrame($frame->getPath());
            } catch (NoSuchFrameException|StaleElementReferenceException $e) {
                continue;
            }

            $frameIdKey = implode('_', $frame->getPath());

            $frameSubDir = sprintf('%s/%s', $subDir, 'frame_' . $frameIdKey);

            $this->createDirIfNotExists($frameSubDir);

            $frameFilename = sprintf('%s_%s', time(), uniqid('', true));
            $framePageContentPath = $frameSubDir . '/' . $frameFilename . '.html';
            $frameAjaxResponsesPath = $frameSubDir . '/' . $frameFilename . '_ajax_responses.json';

            file_put_contents($framePageContentPath, trim($this->driver->getContent()));

            $frameAjaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
            file_put_contents($frameAjaxResponsesPath, json_encode($frameAjaxResponses, JSON_THROW_ON_ERROR));
        }

        $this->framesHandler->switchToMainWindow();

        $zipPath = $this->dumpDir . '/' . uniqid('dump_', true) . '.zip';
        $this->archive($zipPath, $subDir);
        $this->deleteDirectory($subDir);

        return [
            'url' => $this->driver->getCurrentUrl(),
            'dump' => $zipPath,
        ];
    }

    private function deleteDirectory(string $directory): void
    {
        $files = array_diff(scandir($directory), ['.', '..']);
        foreach ($files as $file) {
            (is_dir("$directory/$file")) ? $this->deleteDirectory("$directory/$file") : unlink("$directory/$file");
        }
        rmdir($directory);
    }

    private function archive(string $zipPath, string $subDir): void
    {
        $zip = new ZipArchive();
        if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
            $files = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($subDir));

            foreach ($files as $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($subDir) + 1);
                    $zip->addFile($filePath, $relativePath);
                }
            }

            $zip->close();
        }
    }

    private function createDirIfNotExists(string $dir): void
    {
        if (file_exists($dir)) {
            return;
        }

        if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException(sprintf('Directory "%s" was not created', $dir));
        }
    }
}
