<?php

namespace Avodel\WebDriver\DependencyInjection;

use Avodel\WebDriver\Components\CaptchaVerifier\Solver\CloudflareTurnstileTokenSolverInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\RecaptchaV2SolverInterface;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\CloudflareTurnstileVerifier;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\HCaptchaCaptchaVerifier;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\RecaptchaV2CaptchaVerifier;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Avodel\WebDriver\Extension\WebDriverExtensionInterface;
use Avodel\WebDriver\WebDriver;
use Facebook\WebDriver\Chrome\ChromeOptions;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\Argument\TaggedIteratorArgument;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Extension\Extension;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;

class AvodelWebDriverExtension extends Extension
{
    public function load(array $configs, ContainerBuilder $container)
    {
        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $loader = new YamlFileLoader($container, new FileLocator(__DIR__ . '/../Resources/config'));
        $loader->load('services.yaml');

        $container->registerForAutoconfiguration(WebDriverExtensionInterface::class)->addTag('avodel.web_driver_extension');

        $chromeOptions = new Definition(ChromeOptions::class);
        $chromeOptions->addMethodCall('addArguments', [[
            '--start-maximized',
            '--user-agent=' . $config['user_agent'],
            '--window-size=1920,1080',
            '--disable-blink-features=AutomationControlled',
            '--lang=en-US',
            '--no-sandbox',
        ]]);

        $chromeOptions->addMethodCall('addExtensions', [[
//            __DIR__ . '/../Resources/chrome_extensions/i_don_t_care_about_cookies.crx',
        ]]);

        $desiredCapabilities = [
            'acceptSslCerts' => true,
            ChromeOptions::CAPABILITY => $chromeOptions,
        ];

        $minkWebDriver = new Definition(MinkPhpWebDriver::class);
        $minkWebDriver->setAutowired(true);
        $minkWebDriver->setArguments([
            '$wdHost' => $config['host'],
            '$desiredCapabilities' => $desiredCapabilities,
            '$proxyUrl' => $config['proxy'],
            '$webDriverExtensions' => new TaggedIteratorArgument('avodel.web_driver_extension'),
        ]);

        $webDriver = new Definition(WebDriver::class);
        $webDriver->setAutowired(true);
        $webDriver->setArguments([
            '$driver' => $minkWebDriver,
            '$captchaVerifiers' => [
                $container->getDefinition(RecaptchaV2CaptchaVerifier::class),
                $container->getDefinition(HCaptchaCaptchaVerifier::class),
                $container->getDefinition(CloudflareTurnstileVerifier::class),
            ],
            '$debuggerDumpDir' => $config['components']['debug']['dump_dir'],
            '$maxAjaxWaitingTimeMs' => $config['components']['ajax']['max_waiting_time_ms'],
        ]);

        $container->setDefinition(WebDriver::class, $webDriver);

        $solverClasses = $config['components']['captcha_verifier']['solvers'];
        $container->setAlias(HCaptchaSolverInterface::class, $solverClasses['hcaptcha']);
        $container->setAlias(RecaptchaV2SolverInterface::class, $solverClasses['recaptcha_v2']);
        $container->setAlias(CloudflareTurnstileTokenSolverInterface::class, $solverClasses['cloudflare_turnstile']);

        $container->setParameter('avodel.web_driver.user_agent', $config['user_agent']);
    }
}
