<?php

namespace Avodel\WebDriver\Driver;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\CaptchaVerifier\Verifier\CaptchaVerifier;
use Avodel\WebDriver\Components\Debug\Debugger;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Profile\Profile;
use Behat\Mink\Element\DocumentElement;
use Behat\Mink\Session;
use RuntimeException;
use function get_class;

final readonly class WebDriver
{
    public function __construct(
        private Session $session,
        private AjaxHandler $ajaxHandler,
        private CaptchaVerifier $captchaVerifier,
        private FramesHandler $framesHandler,
        private Debugger $debugger,
    )
    {
    }

    public function __destruct()
    {
        if ($this->session->isStarted()) {
            $this->session->stop();
        }
    }

    public function start(): void
    {
        $this->session->start();
    }

    public function isStarted(): bool
    {
        return $this->session->isStarted();
    }

    public function stop(): void
    {
        $this->session->stop();
    }

    public function visit(string $url): void
    {
        $this->session->visit($url);
    }

    public function reload(): void
    {
        $this->session->reload();
    }

    public function getPage(): DocumentElement
    {
        return $this->session->getPage();
    }

    public function wait(int $time, string $condition): bool
    {
        return $this->session->wait($time, $condition);
    }

    public function getAjaxHandler(): AjaxHandler
    {
        return $this->ajaxHandler;
    }

    public function getDebugger(): Debugger
    {
        return $this->debugger;
    }

    public function getFramesHandler(): FramesHandler
    {
        return $this->framesHandler;
    }

    public function getCaptchaVerifier(): CaptchaVerifier
    {
        return $this->captchaVerifier;
    }

    public function executeScript(string $script): void
    {
        $this->session->executeScript($script);
    }

    public function evaluateScript(string $script): mixed
    {
        return $this->session->evaluateScript($script);
    }

    public function getCurrentUrl(): string
    {
        return $this->session->getCurrentUrl();
    }

    private function getMinkPhpWebDriver(): MinkPhpWebDriver
    {
        $driver = $this->session->getDriver();

        if (!$driver instanceof MinkPhpWebDriver) {
            throw new RuntimeException(sprintf(
                'Expected MinkPhpWebDriver, got "%s"',
                get_class($driver)
            ));
        }

        return $driver;
    }

    public function getTimeSinceLastUserActionMs(): ?int
    {
        return $this->getMinkPhpWebDriver()->getTimeSinceLastUserActionMs();
    }
}
