<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Mouse;

use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;

class Mouse
{
    private const array MOUSE_MOVEMENT_DELAY_MCSEC = ['min' => 9000, 'max' => 11000];
    private const array SCROLL_DELAY_MCSEC = ['min' => 15000, 'max' => 23000];

    private Coordinate $coordinate;

    public function __construct(
        private readonly RemoteWebDriver $webDriver,
        private readonly MousePathStrategyInterface $mousePathStrategy,
    ) {
        $this->coordinate = new Coordinate(random_int(100, 500), random_int(100, 500));
    }

    public function click(Coordinate $coordinate): void
    {
        $devTools = new ChromeDevToolsDriver($this->webDriver);

        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mousePressed',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);
        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mouseReleased',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);
    }

    /**
     * @return void
     */
    public function move(Coordinate $targetCoordinate): void
    {
        $mousePath = $this->mousePathStrategy->calculatePath(
            $this->coordinate,
            $targetCoordinate
        );

        $devTools = new ChromeDevToolsDriver($this->webDriver);
        
        foreach ($mousePath as $coordinate) {
            $devTools->execute('Input.dispatchMouseEvent', [
                'type' => 'mouseMoved',
                'x' => $coordinate->getX(),
                'y' => $coordinate->getY(),
            ]);

            usleep(random_int(
                self::MOUSE_MOVEMENT_DELAY_MCSEC['min'],
                self::MOUSE_MOVEMENT_DELAY_MCSEC['max']
            ));

            $this->coordinate = $coordinate;
        }
    }

    public function scroll(Coordinate $coordinate): void
    {
        $windowHeight = $this->webDriver->manage()->window()->getSize()->getHeight();
        
        $distanceToScroll = $coordinate->getY() - $windowHeight / 2;

        $steps = range(1, ceil(abs($distanceToScroll) / 20));
        $count = count($steps);

        $devTools = new ChromeDevToolsDriver($this->webDriver);
        for ($i = 0; $i < $count; $i++) {
            $deltaY = $steps[$i] * 20;
            if ($distanceToScroll < 0) {
                $deltaY = -$deltaY;
            }

            $devTools->execute('Input.dispatchMouseEvent', [
                'type' => 'mouseWheel',
                'x' => 0,
                'y' => $coordinate->getX(),
                'deltaX' => 0,
                'deltaY' => $deltaY,
            ]);
            usleep(random_int(
                self::SCROLL_DELAY_MCSEC['min'],
                self::SCROLL_DELAY_MCSEC['max']
            ));
        }
    }

    public function getCoordinate(): Coordinate
    {
        return $this->coordinate;
    }
}