<?php

namespace Avodel\WebDriver\UserData;

use InvalidArgumentException;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RuntimeException;
use ZipArchive;

final readonly class UserDataManager
{
    private const array EXPORT_RESOURCES = [
        '/Default/Local Storage/',
        '/Default/Cookies',
        '/Default/History',
    ];

    public function import(string $userDataDirectory, string $userDataBase64): void
    {
        $this->clearDirectory($userDataDirectory);
        file_put_contents($userDataDirectory . '/Default.zip', base64_decode($userDataBase64));
        $this->unzipArchive($userDataDirectory . '/Default.zip');
    }

    private function clearDirectory(string $directory): void
    {
        if (!is_dir($directory)) {
            throw new InvalidArgumentException("The directory '$directory' does not exist.");
        }

        $items = array_diff(scandir($directory), ['.', '..']);

        foreach ($items as $item) {
            $path = $directory . DIRECTORY_SEPARATOR . $item;

            if (is_dir($path)) {
                $this->clearDirectory($path);
                rmdir($path);
            } else {
                unlink($path);
            }
        }
    }

    private function unzipArchive(string $zipFilePath): void
    {
        if (!file_exists($zipFilePath)) {
            throw new InvalidArgumentException('File does not exist.');
        }

        $zip = new ZipArchive();

        if ($zip->open($zipFilePath) === true) {
            $zip->extractTo(dirname($zipFilePath));
            $zip->close();
            return;
        }

        throw new RuntimeException('Could not open archive.');
    }

    public function export(string $userDataDirectory): string
    {
        $tempFile = sys_get_temp_dir() . DIRECTORY_SEPARATOR . uniqid('', true);

        $zip = new ZipArchive();

        if ($zip->open($tempFile, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            throw new RuntimeException('Could not open archive.');
        }

        foreach (self::EXPORT_RESOURCES as $resource) {
            $fullPath = rtrim($userDataDirectory, '/') . $resource;

            if (is_dir($fullPath)) {
                $this->addDirectoryToZip($zip, $fullPath, ltrim($resource, '/'));
            } elseif (is_file($fullPath)) {
                $zip->addFile($fullPath, ltrim($resource, '/'));
            } else {
                throw new RuntimeException('Resource does not exist.');
            }
        }

        $zip->close();

        return file_get_contents($tempFile);
    }

    private function addDirectoryToZip(ZipArchive $zip, string $dirPath, string $zipPath): void
    {
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($dirPath, RecursiveDirectoryIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($iterator as $file) {
            // @phpstan-ignore-next-line
            $localPath = $zipPath . '/' . $iterator->getInnerIterator()->getSubPathName();

            if ($file->isDir()) {
                $zip->addEmptyDir($localPath);
            } else {
                $zip->addFile($file->getRealPath(), $localPath);
            }
        }
    }
}
