<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Mouse;

use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;

class Mouse
{
    private const array MOUSE_MOVEMENT_DELAY_MCSEC = ['min' => 9000, 'max' => 21000];
    private const array SCROLL_DELAY_MCSEC = ['min' => 5000, 'max' => 10000];

    private Coordinate $mouseCoordinate;
    private Coordinate $scrollCoordinate;

    public function __construct(
        private readonly RemoteWebDriver $webDriver,
        private readonly MousePathStrategyInterface $mousePathStrategy,
    ) {
        $this->mouseCoordinate = new Coordinate(random_int(600, 900), random_int(300, 600));
        $this->scrollCoordinate = new Coordinate(0, 0);
    }

    public function click(Coordinate $coordinate): void
    {
        $devTools = new ChromeDevToolsDriver($this->webDriver);

        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mousePressed',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);
        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mouseReleased',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);
    }

    /**
     * @return void
     */
    public function move(Coordinate $targetCoordinate): void
    {
        $mousePath = $this->mousePathStrategy->calculatePath(
            $this->mouseCoordinate,
            $targetCoordinate
        );

        $devTools = new ChromeDevToolsDriver($this->webDriver);
        
        foreach ($mousePath as $coordinate) {
            $devTools->execute('Input.dispatchMouseEvent', [
                'type' => 'mouseMoved',
                'x' => $coordinate->getX(),
                'y' => $coordinate->getY(),
            ]);

            usleep(random_int(
                self::MOUSE_MOVEMENT_DELAY_MCSEC['min'],
                self::MOUSE_MOVEMENT_DELAY_MCSEC['max']
            ));

            $this->mouseCoordinate = $coordinate;
        }
    }

    public function scroll(Coordinate $coordinate): void
    {
        $scrollData = $this->webDriver->executeScript('return {
        scrollTop: window.scrollY,
        scrollHeight: document.documentElement.scrollHeight,
        windowHeight: window.innerHeight
    };');

        $currentScrollPosition = $scrollData['scrollTop'];
        $documentHeight = $scrollData['scrollHeight'];
        $windowHeight = $scrollData['windowHeight'];

        $targetY = $coordinate->getY();
        $deltaY = $targetY - $currentScrollPosition;

        if ($deltaY < 0 && $currentScrollPosition + $deltaY < 0) {
            $deltaY = -$currentScrollPosition;
        }

        if ($currentScrollPosition + $deltaY + $windowHeight >= $documentHeight) {
            $deltaY = $documentHeight - ($currentScrollPosition + $windowHeight);
        }

        $steps = 20;
        $deltaPerStep = $deltaY / $steps;

        $devTools = new ChromeDevToolsDriver($this->webDriver);

        for ($i = 0; $i < $steps; $i++) {
            $devTools->execute('Input.dispatchMouseEvent', [
                'type' => 'mouseWheel',
                'x' => 0,
                'y' => $coordinate->getY(),
                'deltaX' => 0,
                'deltaY' => $deltaPerStep,
            ]);

            usleep(random_int(
                self::SCROLL_DELAY_MCSEC['min'],
                self::SCROLL_DELAY_MCSEC['max']
            ));
        }

        $this->scrollCoordinate = $coordinate;
    }

    public function getMouseCoordinate(): Coordinate
    {
        return $this->mouseCoordinate;
    }

    public function getScrollCoordinate(): Coordinate
    {
        return $this->scrollCoordinate;
    }
}