<?php

namespace Avodel\WebDriver\Extension;

use Avodel\WebDriver\Profile\Profile;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use JsonException;

final readonly class InitFingerprintWebDriverExtension implements WebDriverExtensionInterface
{
    /**
     * @throws JsonException
     */
    public function apply(RemoteWebDriver $webDriver, Profile $profile): void
    {
        $devToolsDriver = new ChromeDevToolsDriver($webDriver);
        $userAgent = $profile->getUserAgent();

        if ($userAgent) {
            $userAgentBrands = [];
            $userAgentMetadata = $userAgent->getUserAgentMetadata();

            foreach ($userAgentMetadata->getBrands() as $brand) {
                $userAgentBrands[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getVersion(),
                ];
            }

            $defaultUserAgentData = [
                'userAgent' => $userAgent->getUserAgent(),
                'platform' => $userAgent->getPlatform(),
                'userAgentMetadata' => [
                    'brands' => $userAgentBrands,
                    'platform' => $userAgentMetadata->getPlatform(),
                    'platformVersion' => $userAgentMetadata->getPlatformVersion(),
                    'architecture' => $userAgentMetadata->getArchitecture(),
                    'model' => $userAgentMetadata->getModel(),
                    'mobile' => $userAgentMetadata->isMobile(),
                ],
            ];

            $devToolsDriver->execute('Emulation.setUserAgentOverride', $defaultUserAgentData);
        }

        $hardwareConcurrency = $profile->getHardwareConcurrency();

        if ($hardwareConcurrency !== null) {
            $devToolsDriver->execute('Emulation.setHardwareConcurrencyOverride', ['hardwareConcurrency' => $hardwareConcurrency]);
        }

        $locale = $profile->getLocale();

        if ($locale !== null) {
            $devToolsDriver->execute('Emulation.setLocaleOverride', ['locale' => $locale]);
        }

        $timezone = $profile->getTimezoneId();

        if ($timezone !== null) {
            $devToolsDriver->execute('Emulation.setTimezoneOverride', ['timezoneId' => $timezone]);
        }

        $deviceMetrics = $profile->getDeviceMetrics();

        if ($deviceMetrics) {
            $devToolsDriver->execute('Emulation.setDeviceMetricsOverride', [
                'width' => $deviceMetrics->getWidth(),
                'height' => $deviceMetrics->getHeight(),
                'deviceScaleFactor' => $deviceMetrics->getDeviceScaleFactor(),
                'mobile' => $deviceMetrics->isMobile(),
                'screenWidth' => $deviceMetrics->getWidth(),
                'screenHeight' => $deviceMetrics->getHeight(),
            ]);
        }

        $preferences = $profile->getPreferences();

        if ($preferences && $preferences->isDarkMode()) {
            $devToolsDriver->execute('Emulation.setEmulatedMedia', [
                'media' => 'screen',
                'features' => [
                    ['name' => 'prefers-color-scheme', 'value' => 'dark'],
                ],
            ]);
        }

        $devToolsDriver->execute('Page.removeScriptToEvaluateOnNewDocument', ['identifier' => '1']);
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'utils.js');

        $deviceMemory = $profile->getDeviceMemory();

        if ($deviceMemory !== null) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.device_memory.js', [$deviceMemory]);
        }

        $mediaDevices = $profile->getMediaDevices();

        if ($mediaDevices) {
            // todo: mask media devices
        }

        $webGl = $profile->getWebGl();

        if ($webGl) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'webgl.vendor.js', [
                $webGl->getRenderer(),
                $webGl->getVendor(),
            ]);
        }

        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hairline.fix.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.webdriver.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hook_remove_cdc_props.js');

        $fonts = $profile->getFonts();

        if ($fonts !== null) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'fonts.js', [$profile->getFonts()]);
        }
    }

    /**
     * @throws JsonException
     */
    private function addScriptToEvaluateOnNewDocument(ChromeDevToolsDriver $devTools, string $source, array $args = []): void
    {
        $source =  '(' . file_get_contents(__DIR__ . '/../Resources/js/fingerprint/' . $source) . ')(' . substr(json_encode($args, JSON_THROW_ON_ERROR), 1, -1) . ');';

        $devTools->execute('Page.addScriptToEvaluateOnNewDocument', [
            'source' => $source,
        ]);
    }
}
