<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Driver\Session;
use Facebook\WebDriver\Exception\NoSuchFrameException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;
use Psr\Log\LoggerInterface;
use RuntimeException;

readonly class DebugDumpFilesPreparer
{
    public function __construct(
        private Session $session,
        private FramesHandler $framesHandler,
        private AjaxHandler $ajaxHandler,
        private LoggerInterface $logger
    ) {
    }

    /**
     * @throws \Exception
     */
    public function prepareDump(string $dir, bool $dumpAssets = false): void
    {
        $this->createDirIfNotExists($dir);

        $filename = sprintf('%s_%s', time(), uniqid('', true));
        $screenshotPath = $dir . '/' . $filename . '.png';
        $pageContentPath = $dir . '/' . $filename . '.html';
        $ajaxResponsesPath = $dir . '/' . $filename . '_ajax_responses.json';
        $pagePdfPath = $dir . '/' . $filename . '.pdf';

        file_put_contents($screenshotPath, $this->session->getScreenshot());
        file_put_contents($pageContentPath, trim($this->session->getPage()->getContent()));

        $ajaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
        file_put_contents($ajaxResponsesPath, json_encode($ajaxResponses, JSON_THROW_ON_ERROR));

        $pdf = $this->session->printToPdf();
        file_put_contents($pagePdfPath, base64_decode($pdf));

        $frames = $this->framesHandler->getFrames();

        foreach ($frames as $frame) {
            try {
                $this->framesHandler->switchToIFrame($frame->getPath());
            } catch (NoSuchFrameException|StaleElementReferenceException $e) {
                continue;
            }

            $frameIdKey = implode('_', $frame->getPath());

            $frameSubDir = sprintf('%s/%s', $dir, 'frame_' . $frameIdKey);

            $this->createDirIfNotExists($frameSubDir);

            $frameFilename = sprintf('%s_%s', time(), uniqid('', true));
            $framePageContentPath = $frameSubDir . '/' . $frameFilename . '.html';
            $frameAjaxResponsesPath = $frameSubDir . '/' . $frameFilename . '_ajax_responses.json';

            file_put_contents($framePageContentPath, trim($this->session->getPage()->getContent()));

            $frameAjaxResponses = $this->ajaxHandler->getAllRawAjaxResponses();
            file_put_contents($frameAjaxResponsesPath, json_encode($frameAjaxResponses, JSON_THROW_ON_ERROR));
        }

        $this->framesHandler->switchToMainWindow();

        if ($dumpAssets) {
            $assetsDir = $dir . '/assets';
            $this->createDirIfNotExists($assetsDir);
            $this->prepareAssets($assetsDir);
        }
    }

    private function prepareAssets(string $dir): void
    {
        $pageSource = $this->session->getPage()->getContent();
        $assetDownloader = new HttpClientAssetDownloader($this->session);

        foreach (['link', 'script', 'img'] as $tag) {
            $elements = $this->session->getPage()->findAll('css', $tag);

            foreach ($elements as $element) {
                $src = $element->getAttribute('href') ?: $element->getAttribute('src');
                if ($src) {
                    $path = $this->downloadAsset($src, $dir, $assetDownloader);
                    if ($path) {
                        $pageSource = str_replace($src, $path, $pageSource);
                    }
                }
            }
        }

        $htmlFile = $dir. '/saved_page.html';
        file_put_contents($htmlFile, $pageSource);
    }

    private function downloadAsset(string $url, string $folder, HttpClientAssetDownloader $assetDownloader): ?string
    {
        $currentUrl = $this->session->getCurrentUrl();
        $parsedCurrent = parse_url($currentUrl);

        $origin = $parsedCurrent['scheme'] . '://' . $parsedCurrent['host'];
        $host = $parsedCurrent['host'];

        $basePath = rtrim(dirname($parsedCurrent['path']), '/');
        $baseUrl = $origin . $basePath;

        if (filter_var($url, FILTER_VALIDATE_URL)) {
            $finalUrl = $url;
        } elseif (str_starts_with($url, '/')) {
            $finalUrl = $origin . $url;
        } else {
            $finalUrl = $baseUrl . '/' . ltrim($url, './');
        }

        $parsedFinal = parse_url($finalUrl);
        $path = $parsedFinal['path'] ?? '';
        $encodedPath = implode('/', array_map('rawurlencode', explode('/', $path)));

        $finalUrl = $parsedFinal['scheme'] . '://' . $parsedFinal['host'];

        if (isset($parsedFinal['port'])) {
            $finalUrl .= ':' . $parsedFinal['port'];
        }

        $finalUrl .= $encodedPath;

        if (!empty($parsedFinal['query'])) {
            $finalUrl .= '?' . $parsedFinal['query'];
        }

        if (!empty($parsedFinal['fragment'])) {
            $finalUrl .= '#' . $parsedFinal['fragment'];
        }

        $allowedExtensions = ['ico', 'jpg', 'jpeg', 'png', 'gif', 'css', 'js', 'woff', 'woff2', 'ttf', 'svg'];
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

        if (!$extension || !in_array($extension, $allowedExtensions, true)) {
            return null;
        }

        if (filter_var($finalUrl, FILTER_VALIDATE_URL)) {
            try {
                $data = $assetDownloader->download($finalUrl);

                $fullPath = rtrim($folder, '/') . '/' . $host . $path;
                $this->createDirIfNotExists(dirname($fullPath));
                file_put_contents($fullPath, $data);

                return './' . $host .  $path;
            } catch (\Throwable $t) {
                $this->logger->warning('Unable to download asset.', ['url' => $finalUrl, 'exc' => $t]);
            }
        }

        return null;
    }

    private function createDirIfNotExists(string $dir): void
    {
        if (file_exists($dir)) {
            return;
        }

        if (!mkdir($dir, 0777, true) && !is_dir($dir)) {
            throw new RuntimeException(sprintf('Directory "%s" was not created', $dir));
        }
    }
}
