<?php

namespace Avodel\WebDriver\Components\Debug;

use Avodel\WebDriver\Driver\Session;
use RuntimeException;

final class HttpClientAssetDownloader
{
    private const array HEADERS = [
        'accept' => '*/*',
        'accept-encoding' => 'gzip, deflate, br, zstd',
        'accept-language' => 'en-US,en;q=0.9',
        'cache-control' => 'no-cache',
        'cookie' => null,
        'pragma' => 'no-cache',
        'priority' => 'u=1',
        'referer' => null,
        'sec-ch-ua' => null,
        'sec-ch-ua-mobile' => '?0',
        'sec-ch-ua-platform' => null,
        'sec-fetch-dest' => null,
        'sec-fetch-mode' => 'cors',
        'sec-fetch-site' => 'same-origin',
        'user-agent' => null,
    ];

    private const array ASSET_EXTENSION_FETCH_DESC_MAP = [
        'ico' => 'image',
        'jpg' => 'image',
        'jpeg' => 'image',
        'png' => 'image',
        'gif' => 'image',
        'css' => 'style',
        'js' => 'script',
        'woff' => 'font',
        'woff2' => 'font',
        'ttf' => 'font',
        'svg' => 'image',
    ];

    private string $currentUrl;
    private string $acceptedLanguages;
    private string $brands;
    private string $cookies;
    private string $userAgent;
    private string $platform;

    public function __construct(private readonly Session $session)
    {
        $this->currentUrl = $session->getCurrentUrl();
        $this->userAgent = $session->evaluateScript('navigator.userAgent');
        $userAgentData = $session->evaluateScript('await navigator.userAgentData.getHighEntropyValues(["architecture", "model", "platformVersion", "uaFullVersion", "fullVersionList", "bitness", "wow64"])');
        $languages = $session->evaluateScript('navigator.languages');
        $this->platform = $userAgentData['platform'];
        $brands = [];

        foreach ($userAgentData['brands'] as $brand) {
            $brands[] = '"' . $brand['brand'] . '";v="' . $brand['version'] . '"';
        }

        $this->brands = implode(', ', $brands);

        $acceptedLanguages = [];

        foreach ($languages as $index => $language) {
            $acceptedLanguages[] = $language . ';q=' . (1 - $index * 0.1);
        }

        array_shift($acceptedLanguages); // remove locale with q=1
        array_unshift($acceptedLanguages, $languages[0]); // push locale to the beginning
        $this->acceptedLanguages = implode(',', $acceptedLanguages);

        $domain = parse_url($this->currentUrl, PHP_URL_HOST);
        $cookies = $session->getAllCookies();
        $cookieValues = [];

        foreach ($cookies as $cookie) {
            if ($cookie->getDomain() === $domain || $cookie->getDomain() === '.' . $domain) {
                $cookieValues[] = $cookie->getName() . '=' . $cookie->getValue();
            }
        }

        $this->cookies = implode('; ', $cookieValues);
    }

    public function download(string $url): string
    {
        $headers = self::HEADERS;

        $headers['cookie'] = $this->cookies;
        $headers['referer'] = $this->currentUrl;
        $headers['sec-ch-ua'] = $this->brands;
        $headers['user-agent'] = $this->userAgent;
        $headers['sec-ch-ua-platform'] = '"' . $this->platform . '"';
        $headers['accept-language'] = $this->acceptedLanguages;

        $extension = strtolower(pathinfo($url, PATHINFO_EXTENSION));
        $fetchDesc = self::ASSET_EXTENSION_FETCH_DESC_MAP[$extension] ?? null;

        if ($fetchDesc) {
            $headers['sec-fetch-dest'] = $fetchDesc;
        }

        $headers = array_filter($headers);

        $proxy = $this->session->getProxy();

        $ch = curl_init($url);

        if ($proxy) {
            curl_setopt($ch, CURLOPT_PROXYPORT, $proxy->getPort());
            curl_setopt($ch, CURLOPT_PROXYTYPE, 'HTTP');
            curl_setopt($ch, CURLOPT_PROXY, $proxy->getHost());
        }

        $formaterHeaders = [];

        foreach ($headers as $header => $value) {
            $formaterHeaders[] = $header . ': ' . $value;
        }

        curl_setopt($ch, CURLOPT_HTTPHEADER, $formaterHeaders);
        curl_setopt($ch, CURLOPT_ENCODING, '');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);

        $data = curl_exec($ch);
        $err = curl_error($ch);
        $errNo = curl_errno($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($errNo !== 0 || !$data || $status >= 400) {
            throw new RuntimeException(sprintf(
                "Failed to download asset from %s. cURL error [%d]: %s. HTTP status: %s",
                $url,
                $errNo,
                $err,
                $status
            ));
        }

        return $data;
    }
}
