<?php

namespace Avodel\WebDriver\Extension;

use Avodel\WebDriver\Profile\Profile;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use JsonException;

final readonly class InitFingerprintWebDriverExtension implements WebDriverExtensionInterface
{
    /**
     * @throws JsonException
     */
    public function apply(RemoteWebDriver $webDriver, Profile $profile): void
    {
        $devToolsDriver = new ChromeDevToolsDriver($webDriver);
        $userAgent = $profile->getUserAgent();

        if ($userAgent) {
            $userAgentBrands = [];
            $userAgentBrandsFullVersion = [];
            $userAgentMetadata = $userAgent->getUserAgentMetadata();

            foreach ($userAgentMetadata->getBrands() as $brand) {
                $userAgentBrands[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getMajorVersion(),
                ];

                $userAgentBrandsFullVersion[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getVersion(),
                ];
            }

            $defaultUserAgentData = [
                'userAgent' => $userAgent->getUserAgent(),
                'platform' => $userAgent->getPlatform(),
                'userAgentMetadata' => [
                    'brands' => $userAgentBrands,
                    'platform' => $userAgentMetadata->getPlatform(),
                    'platformVersion' => $userAgentMetadata->getPlatformVersion(),
                    'architecture' => $userAgentMetadata->getArchitecture(),
                    'model' => $userAgentMetadata->getModel(),
                    'mobile' => $userAgentMetadata->isMobile(),
                ],
            ];

            $devToolsDriver->execute('Emulation.setUserAgentOverride', $defaultUserAgentData);
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.user_agent_data.js', [
                $userAgentBrands,
                $userAgentBrandsFullVersion,
                $userAgent->getFullVersion(),
                $userAgentMetadata->getPlatformVersion(),
                $userAgentMetadata->getArchitecture(),
                $userAgentMetadata->getPlatform(),
            ]);
        }

        $hardwareConcurrency = $profile->getHardwareConcurrency();

        if ($hardwareConcurrency !== null) {
            $devToolsDriver->execute('Emulation.setHardwareConcurrencyOverride', ['hardwareConcurrency' => $hardwareConcurrency]);
        }

        $locale = $profile->getLocale();

        if ($locale !== null) {
            $devToolsDriver->execute('Emulation.setLocaleOverride', ['locale' => $locale]);
        }

        $timezone = $profile->getTimezoneId();

        if ($timezone !== null) {
            $devToolsDriver->execute('Emulation.setTimezoneOverride', ['timezoneId' => $timezone]);
        }

        $deviceMetrics = $profile->getDeviceMetrics();

        if ($deviceMetrics) {
            $scaleFactor = $deviceMetrics->getDeviceScaleFactor();
            $width = (int) ($deviceMetrics->getWidth() / $scaleFactor);
            $height = (int) ($deviceMetrics->getHeight() / $scaleFactor);

            $devToolsDriver->execute('Emulation.setDeviceMetricsOverride', [
                'width' => $width,
                'height' => $height - 85,
                'deviceScaleFactor' => $scaleFactor,
                'mobile' => $deviceMetrics->isMobile(),
                'screenWidth' => $width,
                'screenHeight' => $height,
            ]);
        }

        $preferences = $profile->getPreferences();

        if ($preferences && $preferences->isDarkMode()) {
            $devToolsDriver->execute('Emulation.setEmulatedMedia', [
                'media' => 'screen',
                'features' => [
                    ['name' => 'prefers-color-scheme', 'value' => 'dark'],
                ],
            ]);
        }

        $devToolsDriver->execute('Page.removeScriptToEvaluateOnNewDocument', ['identifier' => '1']);
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'utils.js');

        $deviceMemory = $profile->getDeviceMemory();

        if ($deviceMemory !== null) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.device_memory.js', [$deviceMemory]);
        }

        $webGl = $profile->getWebGl();

        if ($webGl) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'webgl.vendor.js', [
                $webGl->getRenderer(),
                $webGl->getVendor(),
            ]);
        }

        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hairline.fix.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.webdriver.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hook_remove_cdc_props.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hide_cdp.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'window.outerdimensions.js');

        $fonts = $profile->getFonts();

        if ($fonts !== null) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'fonts.js', [$profile->getFonts()]);
        }
    }

    /**
     * @throws JsonException
     */
    private function addScriptToEvaluateOnNewDocument(ChromeDevToolsDriver $devTools, string $source, array $args = []): void
    {
        $source =  '(' . file_get_contents(__DIR__ . '/../Resources/js/fingerprint/' . $source) . ')(' . substr(json_encode($args, JSON_THROW_ON_ERROR), 1, -1) . ');';

        $devTools->execute('Page.addScriptToEvaluateOnNewDocument', [
            'source' => $source,
        ]);
    }
}
