<?php

namespace Avodel\WebDriver\Extension;

use Avodel\WebDriver\Profile\Profile;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use JsonException;

final readonly class InitFingerprintWebDriverExtension implements WebDriverExtensionInterface
{
    /**
     * @throws JsonException
     */
    public function apply(RemoteWebDriver $webDriver, Profile $profile): void
    {
        $devToolsDriver = new ChromeDevToolsDriver($webDriver);
        $devToolsDriver->execute('Page.removeScriptToEvaluateOnNewDocument', ['identifier' => '1']);

        $userAgent = $profile->getUserAgent();

        if ($userAgent) {
            $userAgentBrands = [];
            $userAgentBrandsFullVersion = [];
            $userAgentMetadata = $userAgent->getUserAgentMetadata();

            foreach ($userAgentMetadata->getBrands() as $brand) {
                $userAgentBrands[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getMajorVersion(),
                ];

                $userAgentBrandsFullVersion[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getVersion(),
                ];
            }

            $defaultUserAgentData = [
                'userAgent' => $userAgent->getUserAgent(),
                'platform' => $userAgent->getPlatform(),
                'userAgentMetadata' => [
                    'brands' => $userAgentBrands,
                    'fullVersionList' => $userAgentBrandsFullVersion,
                    'fullVersion' => $userAgent->getFullVersion(),
                    'platform' => $userAgentMetadata->getPlatform(),
                    'platformVersion' => $userAgentMetadata->getPlatformVersion(),
                    'architecture' => $userAgentMetadata->getArchitecture(),
                    'model' => $userAgentMetadata->getModel(),
                    'mobile' => $userAgentMetadata->isMobile(),
                    'bitness' => '64',
                    'wow64' => false,
                ],
            ];

            $devToolsDriver->execute('Emulation.setUserAgentOverride', $defaultUserAgentData);
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.user_agent_data.js', [
                $userAgentBrands,
                $userAgentBrandsFullVersion,
                $userAgent->getFullVersion(),
                $userAgentMetadata->getPlatformVersion(),
                $userAgentMetadata->getArchitecture(),
                $userAgentMetadata->getPlatform(),
            ]);
        }

        if ($hardwareConcurrency = $profile->getHardwareConcurrency()) {
            $devToolsDriver->execute('Emulation.setHardwareConcurrencyOverride', ['hardwareConcurrency' => $hardwareConcurrency]);
        }

        $locale = $profile->getLocale();

        if ($locale !== null) {
            $devToolsDriver->execute('Emulation.setLocaleOverride', ['locale' => $locale]);
        }

        if ($timezone = $profile->getTimezoneId()) {
            $devToolsDriver->execute('Emulation.setTimezoneOverride', ['timezoneId' => $timezone]);
        }

        if ($deviceMetrics = $profile->getDeviceMetrics()) {
            $scaleFactor = $deviceMetrics->getDeviceScaleFactor();
            $width = (int) ($deviceMetrics->getWidth() / $scaleFactor);
            $height = (int) ($deviceMetrics->getHeight() / $scaleFactor);

            $devToolsDriver->execute('Emulation.setDeviceMetricsOverride', [
                'width' => $width,
                'height' => $height - 85,
                'deviceScaleFactor' => $scaleFactor,
                'mobile' => $deviceMetrics->isMobile(),
                'screenWidth' => $width,
                'screenHeight' => $height,
            ]);
        }

        $preferences = $profile->getPreferences();

        if ($preferences && $preferences->isDarkMode()) {
            $devToolsDriver->execute('Emulation.setEmulatedMedia', [
                'media' => 'screen',
                'features' => [
                    ['name' => 'prefers-color-scheme', 'value' => 'dark'],
                ],
            ]);
        }

        if ($deviceMemory = $profile->getDeviceMemory()) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.device_memory.js', [$deviceMemory]);
        }

        if ($webGl = $profile->getWebGl()) {
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'webgl.vendor.js', [
                $webGl->getVendor(),
                $webGl->getRenderer(),
            ]);

            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.gpu.js', [
                strtolower($webGl->getBrand()),
                strtolower($webGl->getArchitecture() ?? ''),
            ]);

            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'canvas.js', [
                $webGl->getRenderer(),
            ]);
        }

        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.webdriver.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hook_remove_cdc_props.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'hide_cdp.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'window.outerdimensions.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'worker.js', [
            $hardwareConcurrency,
            $deviceMemory,
            $userAgent?->getPlatform() ?? 'Win32',
            $webGl?->getVendor(),
            $webGl?->getRenderer(),
        ]);
    }

    /**
     * @throws JsonException
     */
    private function addScriptToEvaluateOnNewDocument(ChromeDevToolsDriver $devTools, string $source, array $args = []): void
    {
        $source =  '(' . file_get_contents(__DIR__ . '/../Resources/js/fingerprint/' . $source) . ')(' . substr(json_encode($args, JSON_THROW_ON_ERROR), 1, -1) . ');';

        $devTools->execute('Page.addScriptToEvaluateOnNewDocument', [
            'source' => $source,
        ]);
    }
}
