<?php

namespace Avodel\WebDriver\Components\CaptchaVerifier\Verifier;

use Avodel\WebDriver\Components\Ajax\AjaxHandler;
use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CaptchaVerificationFailedException;
use Avodel\WebDriver\Components\CaptchaVerifier\Exception\CheckboxFrameNotFoundException;
use Avodel\WebDriver\Components\CaptchaVerifier\Solver\HCaptchaSolverInterface;
use Avodel\WebDriver\Components\Frames\Frame;
use Avodel\WebDriver\Components\Frames\FramesHandler;
use Avodel\WebDriver\Components\Frames\SwitchToFrameException;
use Avodel\WebDriver\Driver\Session;
use Exception;
use Webmozart\Assert\Assert;

final readonly class HCaptchaCaptchaVerifier implements CaptchaVerifierInterface
{
    public function __construct(
        private HCaptchaSolverInterface $hCaptchaSolver,
        private FramesHandler $framesHandler,
        private AjaxHandler $ajaxHandler,
    )
    {
    }

    public function verify(Session $session, array $frames): void
    {
        try {
            $this->doVerify($session, $frames);
        } catch (Exception $e) {
            throw new CaptchaVerificationFailedException('HCaptcha verification failed.', previous: $e);
        } finally {
            $this->framesHandler->switchToMainWindow();
        }
    }

    /**
     * @param array<Frame> $frames
     * @throws CheckboxFrameNotFoundException
     * @throws SwitchToFrameException
     */
    private function doVerify(Session $session, array $frames): void
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session, $frames);

        if (!$checkboxFrame) {
            throw new CheckboxFrameNotFoundException();
        }

        Assert::notNull($challengeFrame, 'Challenge frame was not found.');

        $siteKey = $this->parseQueryValue($challengeFrame->getSrc(), 'sitekey');
        Assert::notEmpty($siteKey, 'Site key was not found.');

        $userAgent = $session->evaluateScript('window.navigator.userAgent');

        $solution = $this->hCaptchaSolver->getSolution(
            $session->getCurrentUrl(),
            $siteKey,
            $userAgent
        );

        $this->framesHandler->switchToIFrame($challengeFrame->getPath());
        $this->ajaxHandler->overrideAjaxResponse(
            'POST',
            'https://api.hcaptcha.com/getcaptcha/' . $siteKey,
            '{"pass": true,"generated_pass_UUID": "' . $solution . '","expiration": 120}'
        );

        $this->framesHandler->switchToIFrame($checkboxFrame->getPath());
        $session->executeScript(file_get_contents(__DIR__ . '/../../../Resources/js/captcha/hcaptcha.js'));

        $session->executeScript('window.checkboxTickPerformed = false');
        $session->getPage()->find('css', '#anchor')->click();

        $checkVerificationStatusScript = <<<JS
window.checkboxTickPerformed === true || window.checkboxTickPerformed === undefined
JS;
        $passed = $session->wait(5000, $checkVerificationStatusScript);
        $session->executeScript('window.checkboxTickPerformed = false');
        Assert::true($passed, 'Checkbox was not checked.');
    }

    public function isVerificationRequired(Session $session, array $frames): bool
    {
        [$checkboxFrame, $challengeFrame] = $this->getHCaptchaFrames($session, $frames);

        return $checkboxFrame && $challengeFrame;
    }

    /**
     * @param array<Frame> $allFrames
     * @return array<Frame|null>
     */
    private function getHCaptchaFrames(Session $session, array $allFrames): array
    {
        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=checkbox')) {
                continue;
            }

            if ($this->isCaptchaSolved($session, $frame)) {
                continue;
            }

            if (!$this->framesHandler->isFrameInteractable($frame->getPath())) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');
            Assert::notNull($currentCaptchaId, 'Captcha id was not found.');

            $challengeFrame = $this->getChallengeFrame($currentCaptchaId, $allFrames);

            return [$frame, $challengeFrame];
        }

        return [null, null];
    }

    /**
     * @param array<Frame> $allFrames
     */
    private function getChallengeFrame(string $captchaId, array $allFrames): Frame
    {
        foreach ($allFrames as $frame) {
            if (!str_starts_with($frame->getSrc(), 'https://newassets.hcaptcha.com/captcha/')) {
                continue;
            }

            if (!str_contains($frame->getSrc(), '#frame=challenge')) {
                continue;
            }

            $currentCaptchaId = $this->parseQueryValue($frame->getSrc(), 'id');

            if ($currentCaptchaId !== $captchaId) {
                continue;
            }

            return $frame;
        }

        throw new \Exception('Challenge frame was not found.');
    }

    private function parseQueryValue(string $src, string $key): ?string
    {
        $parsedUrl = parse_url($src);
        $query = $parsedUrl['fragment'] ?? '';
        parse_str($query, $params);

        return $params[$key] ?? null;
    }

    private function isCaptchaSolved(Session $session, Frame $frame): bool
    {
        $this->framesHandler->switchToIFrame($frame->getPath());
        $isChallengePassed = $session->evaluateScript('window.hCaptchaSolved === true');
        $this->framesHandler->switchToMainWindow();

        return $isChallengePassed;
    }
}
