(function () {
    const originalOpen = XMLHttpRequest.prototype.open;
    const originalSend = XMLHttpRequest.prototype.send;
    const responseOverrides = {};
    const responseHistory = {};
    const activeRequests = {};

    function overrideAjaxResponse(method, url, override, persistent = false) {
        const escapedUrl = getRelativeURL(url).replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const key = `${method.toUpperCase()} ${escapedUrl}`;
        responseOverrides[key] = {
            pattern: new RegExp(`^${escapedUrl}$`),
            override: override,
            persistent: persistent
        };
    }

    function overrideAjaxResponseRegExpr(method, pattern, override, persistent = false) {
        const key = `${method.toUpperCase()} ${pattern}`;
        responseOverrides[key] = {
            pattern: new RegExp(pattern),
            override: override,
            persistent: persistent
        };
    }

    function getAjaxResponses(url) {
        const key = getRelativeURL(url);
        return responseHistory[key] || [];
    }

    function getAllAjaxResponses() {
        return responseHistory;
    }

    function isAjaxRequestActive(url) {
        if (url === undefined) {
            return Object.keys(activeRequests).length > 0;
        }

        const key = getRelativeURL(url);
        return !!activeRequests[key];
    }

    function abortAjaxRequest(url) {
        const key = getRelativeURL(url);
        delete activeRequests[key];
    }

    function getRelativeURL(url) {
        return new URL(url, window.location.origin).pathname + new URL(url, window.location.origin).search;
    }

    XMLHttpRequest.prototype.open = function (method, url) {
        this.url = getRelativeURL(url);
        this.method = method.toUpperCase();
        originalOpen.apply(this, arguments);
    };

    XMLHttpRequest.prototype.send = function (data) {
        const key = `${this.method} ${this.url}`;
        
        let override = null;
        let matchedKey = null;
        
        for (const overrideKey in responseOverrides) {
            const [overrideMethod, overridePattern] = overrideKey.split(' ', 2);
            if (overrideMethod === this.method && responseOverrides[overrideKey].pattern.test(this.url)) {
                override = responseOverrides[overrideKey].override;
                matchedKey = overrideKey;
                break;
            }
        }
        
        const handleResponse = (xhr) => {
            const responseTime = new Date().toISOString();
            const responseDetails = {
                status: xhr.status,
                url: xhr.responseURL || this.url,
                content: xhr.responseText,
                time: responseTime
            };

            if (!responseHistory[this.url]) {
                responseHistory[this.url] = [];
            }
            responseHistory[this.url].push(responseDetails);
        };

        activeRequests[this.url] = true;

        if (override) {
            if (!responseOverrides[matchedKey].persistent) {
                delete responseOverrides[matchedKey];
            }

            Object.defineProperty(this, "response", {writable: true});
            Object.defineProperty(this, "responseText", {writable: true});
            Object.defineProperty(this, "status", {writable: true});
            Object.defineProperty(this, "readyState", {writable: true});

            this.response = override;
            this.responseText = override;
            this.status = 200;
            this.readyState = 4;

            handleResponse(this);

            setTimeout(() => {
                this.dispatchEvent(new Event("readystatechange"));

                if (this.readyState === 4) {
                    this.dispatchEvent(new Event("load"));
                    delete activeRequests[this.url];
                }
            }, 100);

            return;
        }

        this.addEventListener('load', function () {
            if (activeRequests[this.url]) {
                handleResponse(this);
                delete activeRequests[this.url];
            }
        });

        let wasSent = false;

        this.addEventListener('readystatechange', function () {
            if (this.readyState === 2 || this.readyState === 3) {
                wasSent = true;
            }

            if (activeRequests[this.url] && (this.readyState === 4 || (this.readyState === 0 && wasSent))) {
                handleResponse(this);
                delete activeRequests[this.url];
            }
        });

        originalSend.apply(this, arguments);
    };

    window.overrideAjaxResponse = overrideAjaxResponse;
    window.overrideAjaxResponseRegExpr = overrideAjaxResponseRegExpr;
    window.getAjaxResponses = getAjaxResponses;
    window.getAllAjaxResponses = getAllAjaxResponses;
    window.isAjaxRequestActive = isAjaxRequestActive;
    window.abortAjaxRequest = abortAjaxRequest;
    window.ajaxUtilsRegistered = true;
})();
