# Web Driver
* Useful chrome extensions are available out of the box
* Captcha verifiers are available out of the box
* Uses the latest Chrome version
* Uses mouse over and keyboard actions to interact with the browser
* Uses Mink interface to interact with the browser
* Provides a flexible adapter over the main remote web driver.

# Configuration
```yaml
version: '3'

services:
  test-runner:
    image: registry.gitlab.com/mypie/docker-images/frankenphp8.3-cli
    volumes:
      - ./:/app

  browser:
    image: selenium/standalone-chromium:latest
    shm_size: 2gb
```

```yaml
avodel_web_driver:
    host: http://browser:4444/wd/hub
    profile_manager: App\ProfileManager # A profile manager implementation containing logic for making profiles and managing user data
    components:
        debug: ~
        ajax: ~
        captcha_verifier:
            solvers:
                cloudflare_turnstile: App\Context\Login\Captcha\CloudflareTurnstileTokenSolver
```

## Using

```php

use Avodel\WebDriver\Driver\WebDriver;

class Service
{
    public function __construct(
        private readonly WebDriver $webDriver,
    ) {
    }

    public function execute(): void
    {
        $this->webDriver->visit('https://www.vfsglobal.com/');
        $this->webDriver->getDebugger()->dump();
        $allAjaxResponses = $this->webDriver->getAjaxHandler()->getAllAjaxResponses();
        $frames = $this->webDriver->getFramesHandler()->getFrames();
        $this->webDriver->getCaptchaVerifier()->verify();
    }
}
```

# Extensions
Extensions enable the configuration of the WebDriver by applying additional settings, such as:
* Stealth mode
* Dynamic fingerprints
* Ajax overrider component
* Cloudflare Turnstile solver
* HCaptcha solver
* Recaptcha V2 solver

# Components
## Ajax
The Ajax component is a tool for handling asynchronous requests and responses in the Avodel WebDriver. By using the AjaxHandler class, you can fetch, override, and wait for Ajax responses.

Usage:
```php
$ajaxHandler = $this->webDriver->getAjaxHandler();
$ajaxHandler->overrideAjaxResponse('POST', 'https://google.com', '{"key": "value"}');
// ..
```

## Frames
The Frames component is a tool for handling frames and iframes within a web page in the Avodel WebDriver. By using the FramesHandler class, you can get, switch to, and check the interactability of frames.

Usage:
```php
$framesHandler = $this->webDriver->getFramesHandler();
$frames = $framesHandler->getFrames();
// ..
```

## CaptchaVerifier
The CaptchaVerifier component is a tool for verifying captchas on web pages in the Avodel WebDriver. By using the CaptchaVerifier class, you can verify captchas and check if verification is required.

Usage:
```php
$captchaVerifier = $this->webDriver->getCaptchaVerifier();
$captchaVerifier = $framesHandler->verify();
// ..
```

## Debug
By using the Debugger class, you can dump the pages state including the content, frames, and ajax responses and screenshots to a directory.

Usage:
```php
$debugger = $this->webDriver->getDebugger();
$debugger = $debugger->dump();
// ..
```

## How to record a session video

1. Add the Selenium video service to `compose.yaml`:
    ```yaml
   ...
    video:
        image: selenium/video:ffmpeg-7.1-20250101
        volumes:
            - ./.videos:/videos # Directory to store recorded videos
        depends_on:
            - browser
        environment:
            DISPLAY_CONTAINER_NAME: browser # Name of the Selenium browser service in Docker Compose
            FILE_NAME: video.mp4
            FRAME_RATE: 60
   ...
    ```
2. Start all services:
    ```bash
    docker compose up -d
    ```
3. Run your tests.

4. Stop the video service. Once the video service is stopped, the recording will be saved to `.videos/video.mp4`.

### How it Works:
When the video service is running, it automatically connects to the Selenium browser and records the session.  
The recording stops, and the video is saved when the video service is stopped.  

## Real-time Browser Viewing with noVNC

The browser container includes noVNC for real-time viewing through your web browser.

```yaml
    browser:
        image: selenium/standalone-chromium:134.0
        ports:
          - "7900:7900"
```

### Access

Open `http://localhost:7900` in your browser (password: `secret`)

### Features

- Live view with minimal latency
- Mouse & keyboard control for debugging
- No additional software required
