# Avodel Web Driver

A Symfony bundle providing a high-level abstraction over Selenium WebDriver for browser automation with human-like behavior. It solves the problem of programmatically controlling web browsers while mimicking realistic user interactions to bypass bot detection systems.

The bundle wraps Selenium WebDriver and Behat Mink, adding capabilities like native mouse movements via Chrome DevTools Protocol, realistic keyboard input with typo simulation, AJAX interception, captcha solving, and iFrame handling.

## Business Logic

### Core Concept

Web automation tools are often detected by anti-bot systems that analyze behavioral patterns. This bundle addresses this by:

1. **Simulating Human Input** - Mouse movements follow Bezier curves, typing includes realistic delays and occasional typos
2. **Managing Browser State** - Profiles persist sessions, cookies, and fingerprints across restarts
3. **Handling Modern Web Complexity** - Intercepts AJAX, navigates iFrames, and solves captchas automatically

### Main Process

```
1. SESSION INITIALIZATION
   │
   ├── ProfileManager creates Profile (proxy, user agent, device metrics)
   ├── WebDriverFactory builds components (Mouse, Keyboard, Ajax, Frames, Captcha, Debug)
   └── Extensions apply modifications (fingerprinting, stealth, AJAX overrider)
           │
           ▼
2. BROWSER INTERACTION
   │
   ├── visit(url) → Navigate to page
   ├── getMouse() → Human-like movements via Chrome DevTools Protocol
   │   ├── move(coordinate) → Bezier curve path
   │   ├── click(coordinate) → Realistic press/release timing
   │   └── scroll(coordinate) → Batched wheel events with pauses
   ├── getPage() → Mink DocumentElement for DOM queries
   └── executeScript() → Run JavaScript
           │
           ▼
3. COMPONENT OPERATIONS
   │
   ├── AjaxHandler.waitForAjaxResponse(path) → Block until XHR completes
   ├── FramesHandler.switchToIFrame(frame) → Navigate nested frames
   ├── CaptchaVerifier.verify() → Auto-detect and solve captchas
   └── Debugger.dump() → Capture page state (HTML, screenshots, AJAX)
           │
           ▼
4. SESSION CLEANUP
   │
   └── stop() → Close browser, ProfileManager cleans up user data
```

### Mouse Movement Strategy

| Strategy | Behavior | Use Case |
|----------|----------|----------|
| `BasicMousePathStrategy` | Bezier curve interpolation with random deviations | Default - mimics human hand movement |
| `InstantMousePathStrategy` | Direct coordinate jump | Performance testing, no stealth needed |

### Keyboard Typing Simulation

```
Character Input Flow:
─────────────────────
For each character in text:
    │
    ├── Roll typo chance (based on accuracy config)
    │   └── If typo: type wrong chars → pause → backspace → pause
    │
    ├── Calculate delay (60000ms / CPM + random variance)
    │   ├── 60% chance: base delay ± 30%
    │   ├── 20% chance: slower (2-3x base)
    │   └── 20% chance: faster (10% of base)
    │
    └── Dispatch via Chrome DevTools Protocol:
        rawKeyDown → keyDown → char → keyUp
```

### AJAX Interception

The bundle injects JavaScript to intercept all XHR/fetch requests:

| Method | Purpose |
|--------|---------|
| `waitForAjaxResponse(path)` | Block execution until endpoint responds |
| `getAllAjaxResponses()` | Get all captured requests/responses |
| `overrideAjaxResponse(method, url, body)` | Mock responses for testing |
| `getLastDeserializedResponse(pattern, class)` | Parse JSON response into DTO |

### Captcha Solving

```
CaptchaVerifier.verify()
    │
    ├── Scan page for captcha frames (Turnstile, hCaptcha, reCAPTCHA)
    ├── Identify solver from config
    ├── Execute solver implementation
    └── Return success/failure
```

Supported captcha types:
- Cloudflare Turnstile
- hCaptcha
- reCAPTCHA v2

## Architecture

```
WebDriver (facade)
├── Session                    - Low-level browser control
│   └── MinkPhpWebDriver       - Mink/Selenium bridge
├── Mouse                      - CDP-based mouse control
│   └── MousePathStrategy      - Movement interpolation
├── AjaxHandler                - XHR/fetch interception
├── FramesHandler              - iFrame navigation
├── CaptchaVerifier            - Auto captcha solving
│   └── Solver implementations - Per-captcha-type logic
└── Debugger                   - State capture (file/S3)
```

See `src/` for implementation details.

## External Dependencies

| Service | Purpose | Protocol |
|---------|---------|----------|
| Selenium Server | Browser orchestration | WebDriver HTTP API |
| AWS S3 (optional) | Debug dump storage | AWS SDK |
| Captcha Solving Service | Token generation | Custom (per solver) |

## Bundle Configuration

```yaml
avodel_web_driver:
    profile_manager: App\ProfileManager    # ProfileManagerInterface implementation
    auto_accept_cookies: false             # Auto-dismiss cookie banners

    timeouts:
        implicit: 0                        # Element wait (seconds)
        page_load: null                    # Page load timeout
        script: 30                         # Script execution timeout

    mouse:
        enable_tracker: false              # Track mouse position in browser
        path_strategy: BasicMousePathStrategy::class

    components:
        debug: file:///app/.debug          # or s3://bucket/path
        ajax:
            max_waiting_time_ms: 80000     # AJAX response timeout
        captcha_verifier:
            solvers:
                hcaptcha: null             # HCaptchaSolverInterface
                recaptcha_v2: null         # RecaptchaV2SolverInterface
                cloudflare_turnstile: null # CloudflareTurnstileSolverInterface

    typing_accuracy_config:
        accuracy: 1.0                      # 0.0-1.0 (1.0 = no typos)
        on_typo_delay:
            min: 500                       # ms before correction
            max: 1500
        backspace_delay:
            min: 100                       # ms between backspaces
            max: 200
        after_typo_fixed_delay:
            min: 200                       # ms after correction
            max: 600

    extensions: []                         # WebDriverExtensionInterface implementations
```

## Installation

```bash
composer require avodel/web-driver
```

Docker Compose setup:

```yaml
services:
  browser:
    image: selenium/standalone-chromium:latest
    shm_size: 2gb
    ports:
      - "7900:7900"  # noVNC viewer (password: secret)
```

## Usage

```php
use Avodel\WebDriver\Driver\WebDriver;

class AutomationService
{
    public function __construct(private WebDriver $webDriver) {}

    public function execute(): void
    {
        $this->webDriver->visit('https://example.com');

        // Human-like interaction
        $mouse = $this->webDriver->getMouse();
        $mouse->scrollToElement($element);
        $mouse->move(new Coordinate($x, $y));
        $mouse->click(new Coordinate($x, $y));

        // Wait for AJAX
        $ajax = $this->webDriver->getAjaxHandler();
        $ajax->waitForAjaxResponse('/api/data');
        $response = $ajax->getLastDeserializedResponse('#/api/data#', DataDto::class);

        // Handle captcha
        $this->webDriver->getCaptchaVerifier()->verify();

        // Debug dump
        $this->webDriver->getDebugger()->dump();
    }
}
```

## Video Recording

Add Selenium video service:

```yaml
video:
  image: selenium/video:ffmpeg-7.1-20250101
  volumes:
    - ./.videos:/videos
  depends_on:
    - browser
  environment:
    DISPLAY_CONTAINER_NAME: browser
    FILE_NAME: video.mp4
```

Stop the video service to finalize recording.

## Real-time Browser Viewing

Access noVNC at `http://localhost:7900` (password: `secret`) for live browser view with mouse/keyboard control.

## Configuration Variables

| Variable | Description | Default |
|----------|-------------|---------|
| `SELENIUM_HOST` | WebDriver server URL | `http://browser:4444/wd/hub` |
| Debug DSN | Storage location | `file:///app/.debug` |
| Profile Manager | Session/profile handler | `FallbackProfileManager` |
