<?php

namespace Avodel\WebDriver\Components\Ajax;

use Avodel\WebDriver\Components\Ajax\Exception\AjaxResponseWasNotReceivedException;
use Avodel\WebDriver\Components\Ajax\Exception\DeserializeResponseException;
use Avodel\WebDriver\Components\Ajax\Exception\UnsuccessfulAjaxResponseException;
use Avodel\WebDriver\Driver\Session;
use Psr\Log\LoggerInterface;
use Psr\Clock\ClockInterface;
use Symfony\Component\Serializer\Exception\ExceptionInterface;
use Symfony\Component\Serializer\SerializerInterface;

final class AjaxHandler
{
    private ?\DateTimeImmutable $filterThreshold = null;
    private ?int $lastLogTimestamp = null;

    public function __construct(
        private Session $session,
        private int $maxWaitingTimeMs,
        private LoggerInterface $logger,
        private SerializerInterface $serializer,
        private ClockInterface $clock
    )
    {
    }

    private function register(): void
    {
        $registered = $this->session->evaluateScript('window.ajaxUtilsRegistered');

        if ($registered === true) {
            return;
        }

        $this->session->executeScript(file_get_contents(__DIR__ . '/../../Resources/js/ajaxoverrider.js'));
    }

    /**
     * Overrides an Ajax response with a custom response.
     */
    public function overrideAjaxResponse(string $method, string $url, string $body, bool $persistent = false): void
    {
        $this->register();
        $escapedBody = addslashes($body);
        $escapedBody = str_replace("\n", "\\n", $escapedBody);

        $statement = sprintf('window.overrideAjaxResponse(\'%s\', \'%s\', \'%s\', %s);', $method, $url, $escapedBody, $persistent ? 'true' : 'false');

        $this->session->executeScript($statement);
    }

    /**
     * Returns the last Ajax response for the given path.
     */
    public function getLastAjaxResponse(string $path): ?AjaxResponse
    {
        $this->register();
        $responses = $this->session->evaluateScript('window.getAjaxResponses(\'' . $path . '\')');

        if (!$responses) {
            return null;
        }

        $lastResponse = $responses[array_key_last($responses)];
        $responseTime = new \DateTimeImmutable($lastResponse['time']);
        if(!$this->filterThreshold || $responseTime >= $this->filterThreshold) {
            return new AjaxResponse(
                (string)$lastResponse['url'],
                (int)$lastResponse['status'],
                (string)$lastResponse['content'],
                new \DateTimeImmutable($lastResponse['time']),
            );
        }

        return null;
    }

    /**
     * Returns the last Ajax response matching a specific search pattern with ability to search by full url.
     */
    public function getLastResponse(string $pathPattern, $fullUrlSearch = false): ?AjaxResponse
    {
        $this->register();
        $ajaxResponses = $this->getAllAjaxResponses();

        if (!$ajaxResponses) {
            return null;
        }

        $foundResponses = [];

        foreach ($ajaxResponses as $response) {
            $requestUrlData = parse_url($response->getUrl());
            $requestPath = $requestUrlData['path'] ?? '';

            $target = $fullUrlSearch ? $response->getUrl() : $requestPath;

            if (preg_match($pathPattern, $target)) {
                if (!$this->filterThreshold || $response->getTime() >= $this->filterThreshold) {
                    $foundResponses[] = $response;
                }
            }
        }

        return count($foundResponses) ? end($foundResponses) : null;
    }

    /**
     * @template T of object
     * @phpstan-template T of object
     *
     * @param class-string<T>|string $className
     *
     * @return T|mixed
     * @phpstan-return T|mixed
     *
     * @throws UnsuccessfulAjaxResponseException
     * @throws DeserializeResponseException
     *
     * @phpstan-ignore method.templateTypeNotInParameter
     */
    public function getLastDeserializedResponse(string $pathPattern, string $className, string $format = 'json'): mixed
    {
        $response = $this->getLastResponse($pathPattern);

        if (!$response) {
            return null;
        }

        $content = $response->getContent();

        if ($response->getStatusCode() < 200 || $response->getStatusCode() >= 300) {
            $this->logger->debug('Unsuccessful ajax response.', [
                'path' => $pathPattern,
                'content' => substr($content, 0, 2000),
                'status' => $response->getStatusCode(),
            ]);

            throw new UnsuccessfulAjaxResponseException('The AJAX response was not successful.');
        }

        try {
            return $this->serializer->deserialize($content, $className, $format);
        } catch (ExceptionInterface $e) {
            $this->logger->error('Unable to deserialize AJAX response.', [
                'path' => $pathPattern,
                'content' => substr($content, 0, 2000),
                'format' => $format,
                'class' => $className,
                'exception' => $e,
            ]);

            throw new DeserializeResponseException('Unable to deserialize ajax response.', 0, $e);
        }
    }

    /**
     * Returns all raw Ajax responses.
     */
    public function getAllRawAjaxResponses(): array
    {
        $this->register();
        $executable = <<<JS
return typeof window.getAllAjaxResponses === 'function' ? window.getAllAjaxResponses() : [];
JS;

        return $this->session->evaluateScript($executable);
    }

    /**
     * Returns all Ajax responses.
     *
     * @return AjaxResponse[]
     */
    public function getAllAjaxResponses(): array
    {
        $endpointResponses = $this->getAllRawAjaxResponses();
        $result = [];

        foreach ($endpointResponses as $responses) {
            foreach ($responses as $response) {
                $result[] = new AjaxResponse(
                    (string)$response['url'],
                    (int)$response['status'],
                    (string)$response['content'],
                    new \DateTimeImmutable($response['time']),
                );
            }
        }

        return $result;
    }

    /**
     * Waits for an Ajax response to complete for the given path.
     *
     * @throws AjaxResponseWasNotReceivedException in case the AJAX response was not received
     */
    public function waitForAjaxResponse(string $path): void
    {
        $this->register();
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);
            $this->abortFailedRequests();

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The XHR request was not finished.', [
                    'path' => $path,
                    'executedTimeMs' => $executedTimeMs,
                ]);

                $this->session->executeScript(sprintf('window.abortAjaxRequest("%s")', $path));

                throw new AjaxResponseWasNotReceivedException(sprintf('The AJAX response was not received. Path: %s', $path));
            }

            $isLoadingActive = $this->session->evaluateScript(sprintf('window.isAjaxRequestActive("%s")', $path));

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    /**
     * Waits until all Ajax requests are finished.
     *
     * @throws AjaxResponseWasNotReceivedException in case the AJAX response was not received
     */
    public function waitUntilAllAjaxRequestsAreFinished(): void
    {
        $this->register();
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);
            $this->abortFailedRequests();

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The AJAX request was not finished.', [
                    'executedTimeMs' => $executedTimeMs,
                ]);

                throw new AjaxResponseWasNotReceivedException('The AJAX response was not received.');
            }

            $isLoadingActive = $this->session->evaluateScript('window.isAjaxRequestActive()');

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    public function clearHistory(): void
    {
        $this->filterThreshold = $this->clock->now();
    }

    /**
     * Checks if an Ajax request is active.
     */
    public function isAjaxRequestActive(): bool
    {
        $this->register();
        return $this->session->evaluateScript('window.isAjaxRequestActive()');
    }

    private function abortFailedRequests(): void
    {
        $logs = $this->session->getBrowserLogs();

        foreach ($logs as $log) {
            if ($log['source'] !== 'network') {
                continue;
            }

            if ($this->lastLogTimestamp && $this->lastLogTimestamp > $log['timestamp']) {
                continue;
            }

            $this->lastLogTimestamp = $log['timestamp'];
            preg_match('#https?://[^/]+(?P<path>/[^\s]+).*?status of (?P<code>\d{3})#', $log['message'], $matches);

            $path = $matches['path'] ?? null;

            if ($path) {
                $this->session->executeScript(sprintf('window.abortAjaxRequest("%s")', $path));
            }
        }
    }
}
