<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Components\Debug;

use Aws\S3\S3Client;

final readonly class DebugDumperFactory
{
    public function __construct(private string $dumpStorageDsn)
    {
    }

    public function create(): DebugDumperInterface
    {
        switch (true) {
            case str_starts_with($this->dumpStorageDsn, 's3://'):
                $s3Config = parse_url($this->dumpStorageDsn);

                $awsKey = urldecode($s3Config['user'] ?? '');
                $awsSecret = urldecode($s3Config['pass'] ?? '');
                $region = urldecode($s3Config['host'] ?? '');

                $path = isset($s3Config['path']) ? ltrim($s3Config['path'], '/') : '';

                $bucketName = strtok($path, '/');
                $pathPrefix = substr($path, strlen($bucketName) + 1);

                if (in_array(null, [$awsKey, $awsSecret, $bucketName, $region])) {
                    throw new \InvalidArgumentException("Invalid S3 config");
                }

                if (!class_exists(S3Client::class)) {
                    throw new \RuntimeException('The Aws\S3\S3Client class is not installed. Please run "composer require aws/aws-sdk-php" if you want to use S3 debug storage.');
                }

                $s3Client = new S3Client(
                    args: [
                        'region' => $region,
                        'version' => 'latest',
                        'credentials' => [
                            'key' => $awsKey,
                            'secret' => $awsSecret,
                        ],
                    ]
                );

                return new S3DebugDumper($s3Client, $bucketName, $pathPrefix);
            case str_starts_with($this->dumpStorageDsn, 'file://'):
                $localPath = parse_url($this->dumpStorageDsn, PHP_URL_PATH);

                return new LocalDebugDumper($localPath);
            default:
                throw new \InvalidArgumentException("Unsupported DSN format: {$this->dumpStorageDsn}");
        }
    }
}
