<?php

namespace Avodel\WebDriver\Components\Debug;

use Aws\S3\S3Client;
use ZipArchive;

readonly class S3DebugDumper implements DebugDumperInterface
{
    public function __construct(
        private S3Client $s3Client,
        private string $bucket,
        private string $pathPrefix = '',
    ) {
    }

    public function save(DebugDumpFilesPreparer $debugDumpFilesPreparer, DumpOpts $dumpOpts): string
    {
        $dumpName = $dumpOpts->getDumpName();
        $subDir = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $dumpName;
        $debugDumpFilesPreparer->prepareDump($subDir, $dumpOpts->isDumpAssets());

        $dumpName .= '.zip';

        $zipPath = $subDir . '/' . $dumpName;
        $this->archiveAsZip($zipPath, $subDir);

        $key = '';

        if ($this->pathPrefix) {
            $key .= $this->pathPrefix . '/';
        }

        $key .= $dumpName;

        $result = $this->s3Client->putObject([
            'Bucket' => $this->bucket,
            'Key' => $key,
            'Body' => file_get_contents($zipPath),
        ]);

        $this->deleteDirectory($subDir);

        return $result['ObjectURL'];
    }

    private function deleteDirectory(string $directory): void
    {
        $files = array_diff(scandir($directory), ['.', '..']);
        foreach ($files as $file) {
            (is_dir("$directory/$file")) ? $this->deleteDirectory("$directory/$file") : unlink("$directory/$file");
        }
        rmdir($directory);
    }

    private function archiveAsZip(string $zipPath, string $dirToArchive): void
    {
        $zip = new ZipArchive();

        if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
            $files = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($dirToArchive));

            foreach ($files as $file) {
                if (!$file->isDir()) {
                    $filePath = $file->getRealPath();
                    $relativePath = substr($filePath, strlen($dirToArchive) + 1);
                    $zip->addFile($filePath, $relativePath);
                }
            }

            $zip->close();
        }
    }
}