<?php

namespace Avodel\WebDriver\Extension;

use Avodel\WebDriver\Profile\Profile;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use JsonException;

final readonly class InitFingerprintWebDriverExtension implements WebDriverExtensionInterface
{
    /**
     * @throws JsonException
     */
    public function apply(RemoteWebDriver $webDriver, Profile $profile): void
    {
        $devToolsDriver = new ChromeDevToolsDriver($webDriver);
        // removes cdc_* objects
        $devToolsDriver->execute('Page.removeScriptToEvaluateOnNewDocument', ['identifier' => '1']);

        $userAgent = $profile->getUserAgent();

        if ($userAgent) {
            $userAgentBrands = [];
            $userAgentBrandsFullVersion = [];
            $userAgentMetadata = $userAgent->getUserAgentMetadata();

            foreach ($userAgentMetadata->getBrands() as $brand) {
                $userAgentBrands[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getMajorVersion(),
                ];

                $userAgentBrandsFullVersion[] = [
                    'brand' => $brand->getBrand(),
                    'version' => $brand->getVersion(),
                ];
            }

            $defaultUserAgentData = [
                'userAgent' => $userAgent->getUserAgent(),
                'platform' => $userAgent->getPlatform(),
                'userAgentMetadata' => [
                    'brands' => $userAgentBrands,
                    'fullVersionList' => $userAgentBrandsFullVersion,
                    'fullVersion' => $userAgent->getFullVersion(),
                    'platform' => $userAgentMetadata->getPlatform(),
                    'platformVersion' => $userAgentMetadata->getPlatformVersion(),
                    'architecture' => $userAgentMetadata->getArchitecture(),
                    'model' => '',
                    'mobile' => false,
                    'bitness' => '64',
                    'wow64' => false,
                ],
            ];

            $devToolsDriver->execute('Emulation.setUserAgentOverride', $defaultUserAgentData);
            $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.user_agent_data.js', [
                $userAgentBrands,
                $userAgentBrandsFullVersion,
                $userAgent->getFullVersion(),
                $userAgentMetadata->getPlatformVersion(),
                $userAgentMetadata->getArchitecture(),
                $userAgentMetadata->getPlatform(),
            ]);
        }

        $locale = $profile->getLocale();

        if ($locale !== null) {
            $devToolsDriver->execute('Emulation.setLocaleOverride', ['locale' => $locale]);
        }

        if ($timezone = $profile->getTimezoneId()) {
            $devToolsDriver->execute('Emulation.setTimezoneOverride', ['timezoneId' => $timezone]);
        }

        if ($deviceMetrics = $profile->getDeviceMetrics()) {
            $scaleFactor = $deviceMetrics->getDeviceScaleFactor();
            $width = (int) ($deviceMetrics->getWidth() / $scaleFactor);
            $height = (int) ($deviceMetrics->getHeight() / $scaleFactor);

            $devToolsDriver->execute('Emulation.setDeviceMetricsOverride', [
                'width' => $width,
                'height' => $height - 125,
                'deviceScaleFactor' => $scaleFactor,
                'mobile' => $deviceMetrics->isMobile(),
                'screenWidth' => $width,
                'screenHeight' => $height,
            ]);
        }

        $preferences = $profile->getPreferences();

        if ($preferences && $preferences->isDarkMode()) {
            $devToolsDriver->execute('Emulation.setEmulatedMedia', [
                'media' => 'screen',
                'features' => [
                    ['name' => 'prefers-color-scheme', 'value' => 'dark'],
                ],
            ]);
        }

        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'navigator.outerdimensions.js');
        $this->addScriptToEvaluateOnNewDocument($devToolsDriver, 'math_warmup.js');
    }

    /**
     * @throws JsonException
     */
    private function addScriptToEvaluateOnNewDocument(ChromeDevToolsDriver $devTools, string $source, array $args = []): void
    {
        $source =  '(' . file_get_contents(__DIR__ . '/../Resources/js/fingerprint/' . $source) . ')(' . substr(json_encode($args, JSON_THROW_ON_ERROR), 1, -1) . ');';

        $devTools->execute('Page.addScriptToEvaluateOnNewDocument', [
            'source' => $source,
        ]);
    }
}
