<?php

declare(strict_types=1);

namespace Avodel\WebDriver\Mouse;

use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use Facebook\WebDriver\WebDriverElement;

class Mouse
{
    private Coordinate $mouseCoordinate;
    private Coordinate $scrollCoordinate;
    private int $tickDeltaY;

    public function __construct(
        private readonly RemoteWebDriver $webDriver,
        private readonly MousePathStrategyInterface $mousePathStrategy,
        private readonly string $platform,
    ) {
        $this->mouseCoordinate = new Coordinate(random_int(600, 900), random_int(300, 600));
        $this->scrollCoordinate = new Coordinate(0, 0);
        $this->tickDeltaY = match ($this->platform) {
            'macOS' => 40,
            'Linux' => 53,
            default => 120
        };
    }

    public function click(Coordinate $coordinate): void
    {
        $devTools = new ChromeDevToolsDriver($this->webDriver);

        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mousePressed',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);

        usleep(random_int(30, 120) * 1000);

        $devTools->execute('Input.dispatchMouseEvent', [
            'type' => 'mouseReleased',
            'x' => $coordinate->getX(),
            'y' => $coordinate->getY(),
            'button' => 'left',
            'clickCount' => 1,
        ]);
    }

    /**
     * @return void
     */
    public function move(Coordinate $targetCoordinate): void
    {
        $mousePath = $this->mousePathStrategy->calculatePath(
            $this->mouseCoordinate,
            $targetCoordinate
        );

        $devTools = new ChromeDevToolsDriver($this->webDriver);

        foreach ($mousePath as $coordinate) {
            $devTools->execute('Input.dispatchMouseEvent', [
                'type' => 'mouseMoved',
                'x' => $coordinate->getX(),
                'y' => $coordinate->getY(),
            ]);

            $this->mouseCoordinate = $coordinate;
        }

        usleep(random_int(100, 250) * 1000);
    }

    public function scroll(Coordinate $coordinate): void
    {
        $scrollData = $this->webDriver->executeScript('return {
        scrollTop: window.scrollY,
        scrollHeight: document.documentElement.scrollHeight,
        windowHeight: window.innerHeight
    };');

        $currentScrollPosition = $scrollData['scrollTop'];
        $documentHeight = $scrollData['scrollHeight'];
        $windowHeight = $scrollData['windowHeight'];

        $targetY = $coordinate->getY();
        $deltaY = $targetY - $currentScrollPosition;

        if ($deltaY < 0 && $currentScrollPosition + $deltaY < 0) {
            $deltaY = -$currentScrollPosition;
        }

        if ($currentScrollPosition + $deltaY + $windowHeight >= $documentHeight) {
            $deltaY = $documentHeight - ($currentScrollPosition + $windowHeight);
        }

        if ($deltaY === 0) {
            return;
        }

        $deltaPerStep = ($deltaY < $this->tickDeltaY && $deltaY > -$this->tickDeltaY) ? $deltaY : (($deltaY < 0) ? -$this->tickDeltaY : $this->tickDeltaY);
        $division = abs($deltaPerStep);
        $steps = 0;

        if ($division > 0) {
            $steps = (int) ceil(abs($deltaY) / abs($deltaPerStep));
        }

        $devTools = new ChromeDevToolsDriver($this->webDriver);

        $currentStep = 0;

        while ($currentStep < $steps) {
            $remainingSteps = $steps - $currentStep;
            $batchSize = min(random_int(4, 10), $remainingSteps);

            for ($i = 0; $i < $batchSize; $i++) {
                $devTools->execute('Input.dispatchMouseEvent', [
                    'type' => 'mouseWheel',
                    'x' => $this->mouseCoordinate->getX(),
                    'y' => $this->mouseCoordinate->getY(),
                    'deltaX' => 0,
                    'deltaY' => $deltaPerStep,
                ]);

                usleep(random_int(10, 80) * 1000);
            }

            usleep(random_int(80, 800) * 1000);

            $currentStep += $batchSize;
        }

        $this->scrollCoordinate = $coordinate;

        usleep(random_int(80, 120) * 1000);
    }

    /**
     * Scroll within a specific container element
     *
     * @param string $containerSelector CSS selector for the scrollable container
     * @param int $deltaY Amount to scroll (positive = down, negative = up)
     * @param Coordinate|null $scrollPosition Optional position to scroll to within container
     */
    public function scrollContainer(string $containerSelector, int $deltaY, ?Coordinate $scrollPosition = null): void
    {
        // First, get container information
        $containerData = $this->webDriver->executeScript("
            const container = document.querySelector(arguments[0]);
            if (!container) {
                return null;
            }
            
            const rect = container.getBoundingClientRect();
            return {
                scrollTop: container.scrollTop,
                scrollHeight: container.scrollHeight,
                clientHeight: container.clientHeight,
                offsetTop: rect.top + window.scrollY,
                offsetLeft: rect.left + window.scrollX,
                width: rect.width,
                height: rect.height,
                hasScroll: container.scrollHeight > container.clientHeight
            };
        ", [$containerSelector]);

        if (!$containerData || !$containerData['hasScroll']) {
            // Container doesn't exist or isn't scrollable, fallback to page scroll
            if ($scrollPosition) {
                $this->scroll($scrollPosition);
            }
            return;
        }

        // Calculate scroll parameters
        $currentScrollTop = $containerData['scrollTop'];
        $scrollHeight = $containerData['scrollHeight'];
        $clientHeight = $containerData['clientHeight'];

        if ($scrollPosition) {
            // Scroll to specific position within container
            $targetScrollTop = $scrollPosition->getY();
            $deltaY = $targetScrollTop - $currentScrollTop;
        }

        // Validate scroll bounds
        $maxScrollTop = $scrollHeight - $clientHeight;
        $targetScrollTop = $currentScrollTop + $deltaY;

        if ($targetScrollTop < 0) {
            $deltaY = -$currentScrollTop;
        } elseif ($targetScrollTop > $maxScrollTop) {
            $deltaY = $maxScrollTop - $currentScrollTop;
        }

        if ($deltaY === 0) {
            return;
        }

        // Calculate mouse position over container center
        $mouseX = (int) ($containerData['offsetLeft'] + $containerData['width'] / 2);
        $mouseY = (int) ($containerData['offsetTop'] + $containerData['height'] / 2);

        // Move mouse to container first
        $this->move(new Coordinate($mouseX, $mouseY));

        // Perform scrolling
        $this->performScrolling($deltaY, new Coordinate($mouseX, $mouseY));
    }

    public function performScrolling(int $deltaY, Coordinate $mousePosition): void
    {
        if ($deltaY === 0) {
            return;
        }

        $deltaPerStep = ($deltaY < $this->tickDeltaY && $deltaY > -$this->tickDeltaY) ?
            $deltaY :
            (($deltaY < 0) ? -$this->tickDeltaY : $this->tickDeltaY);

        $steps = 0;
        $division = abs($deltaPerStep);

        if ($division > 0) {
            $steps = (int) ceil(abs($deltaY) / abs($deltaPerStep));
        }

        $devTools = new ChromeDevToolsDriver($this->webDriver);
        $currentStep = 0;

        while ($currentStep < $steps) {
            $remainingSteps = $steps - $currentStep;
            $batchSize = min(random_int(4, 10), $remainingSteps);

            for ($i = 0; $i < $batchSize; $i++) {
                $devTools->execute('Input.dispatchMouseEvent', [
                    'type' => 'mouseWheel',
                    'x' => $mousePosition->getX(),
                    'y' => $mousePosition->getY(),
                    'deltaX' => 0,
                    'deltaY' => $deltaPerStep,
                ]);

                usleep(random_int(10, 80) * 1000);
            }

            usleep(random_int(80, 800) * 1000);
            $currentStep += $batchSize;
        }

        usleep(random_int(80, 120) * 1000);
    }

    /**
     * Scroll to make a specific WebDriver element visible
     * This method automatically detects if the element is in a scrollable container
     * or if page-level scrolling is needed
     *
     * @param WebDriverElement $element The element to scroll to
     * @param bool $centerElement Whether to center the element in viewport/container (default: true)
     */
    public function scrollToElement(WebDriverElement $element, bool $centerElement = true): void
    {
        // First check if element is already visible
        if ($this->isElementVisible($element)) {
            return;
        }

        $containerInfo = $this->findScrollableContainerForElement($element);

        if ($containerInfo) {
            $this->scrollElementIntoViewInContainer($containerInfo, $centerElement);
        } else {
            $this->scrollElementIntoViewOnPage($element, $centerElement);
        }
    }

    /**
     * Check if element is currently visible in viewport
     */
    private function isElementVisible(WebDriverElement $element): bool
    {
        $script = <<<EOF
const element = arguments[0];
const rect = element.getBoundingClientRect();
const elementAtCenter = document.elementFromPoint(
    rect.left + (rect.width / 2), 
    rect.top + (rect.height / 2)
);
return element.contains(elementAtCenter) || elementAtCenter === element;
EOF;

        return (bool) $this->webDriver->executeScript($script, [$element]);
    }

    /**
     * Find the scrollable container that contains the given element
     */
    private function findScrollableContainerForElement(WebDriverElement $element): ?array
    {
        $script = <<<EOF
const targetElement = arguments[0];
let element = targetElement.parentElement;

// Walk up the DOM tree to find scrollable container
while (element && element !== document.body && element !== document.documentElement) {
    const style = window.getComputedStyle(element);
    const hasVerticalScroll = element.scrollHeight > element.clientHeight;
    const canScroll = style.overflowY === 'scroll' || 
                    style.overflowY === 'auto' || 
                    style.overflow === 'scroll' || 
                    style.overflow === 'auto';
    
    if (hasVerticalScroll && canScroll) {
        const rect = element.getBoundingClientRect();
        const targetRect = targetElement.getBoundingClientRect();
        
        // Generate a unique selector for the container
        let selector = element.tagName.toLowerCase();
        if (element.id) {
            selector = '#' + element.id;
        } else if (element.className) {
            const classes = element.className.trim().split(/\s+/).filter(c => c.length > 0);
            if (classes.length > 0) {
                selector = element.tagName.toLowerCase() + '.' + classes.join('.');
            }
        }
        
        return {
            selector: selector,
            containerRect: {
                top: rect.top,
                left: rect.left,
                width: rect.width,
                height: rect.height
            },
            scrollTop: element.scrollTop,
            scrollHeight: element.scrollHeight,
            clientHeight: element.clientHeight,
            targetRect: {
                top: targetRect.top,
                left: targetRect.left,
                width: targetRect.width,
                height: targetRect.height
            }
        };
    }
    element = element.parentElement;
}

return null;
EOF;

        return $this->webDriver->executeScript($script, [$element]);
    }

    /**
     * Scroll element into view within its scrollable container
     */
    private function scrollElementIntoViewInContainer(array $containerInfo, bool $centerElement): void
    {
        $containerRect = $containerInfo['containerRect'];
        $targetRect = $containerInfo['targetRect'];
        $currentScrollTop = $containerInfo['scrollTop'];
        $containerHeight = $containerInfo['clientHeight'];

        // Calculate relative position of target within container
        $targetRelativeTop = $targetRect['top'] - $containerRect['top'];

        if ($centerElement) {
            // Center the element in the container
            $targetScrollTop = $currentScrollTop + $targetRelativeTop - ($containerHeight / 2) + ($targetRect['height'] / 2);

            // Add some randomization for natural behavior
            $deviation = random_int(-30, 30);
            $targetScrollTop += $deviation;
        } else {
            // Just make element visible (minimal scroll)
            if ($targetRelativeTop < 0) {
                // Element is above visible area
                $targetScrollTop = $currentScrollTop + $targetRelativeTop - 20; // 20px padding
            } elseif ($targetRelativeTop + $targetRect['height'] > $containerHeight) {
                // Element is below visible area
                $targetScrollTop = $currentScrollTop + $targetRelativeTop - $containerHeight + $targetRect['height'] + 20;
            } else {
                // Element is already visible
                return;
            }
        }

        // Ensure we don't scroll beyond bounds
        $maxScrollTop = $containerInfo['scrollHeight'] - $containerInfo['clientHeight'];
        $targetScrollTop = max(0, min($targetScrollTop, $maxScrollTop));

        // Calculate delta from current position
        $deltaY = $targetScrollTop - $currentScrollTop;

        if (abs($deltaY) > 5) { // Only scroll if significant difference
            // Move mouse to container center first
            $mouseX = (int) ($containerRect['left'] + $containerRect['width'] / 2);
            $mouseY = (int) ($containerRect['top'] + $containerRect['height'] / 2);
            $this->move(new Coordinate($mouseX, $mouseY));

            // Perform the scroll
            $this->scrollContainer($containerInfo['selector'], (int)$deltaY);
        }
    }

    /**
     * Scroll element into view on page level
     */
    private function scrollElementIntoViewOnPage(WebDriverElement $element, bool $centerElement): void
    {
        $location = $element->getLocation();

        if ($centerElement) {
            $scrollToY = $this->calculateScrollYWithRandomOffset($element);
        } else {
            // Minimal scroll to make element visible
            $script = <<<EOF
const element = arguments[0];
const rect = element.getBoundingClientRect();
const viewportHeight = window.innerHeight;
const currentScrollY = window.scrollY;

// Check if element is already visible
if (rect.top >= 0 && rect.bottom <= viewportHeight) {
    return currentScrollY; // No scroll needed
}

let targetY;
if (rect.top < 0) {
    // Element is above viewport
    targetY = currentScrollY + rect.top - 50; // 50px padding
} else {
    // Element is below viewport
    targetY = currentScrollY + rect.bottom - viewportHeight + 50;
}

// Ensure bounds
if (targetY < 0) targetY = 0;
const maxScrollY = document.documentElement.scrollHeight - viewportHeight;
if (targetY > maxScrollY) targetY = maxScrollY;

return targetY;
EOF;

            $scrollToY = (int) $this->webDriver->executeScript($script, [$element]);
        }

        $this->scroll(new Coordinate($location->getX(), $scrollToY));
    }

    /**
     * Calculate scroll position with random offset for natural behavior
     */
    private function calculateScrollYWithRandomOffset(WebDriverElement $element): int
    {
        $script = <<<EOF
const element = arguments[0];
const rect = element.getBoundingClientRect();
const viewportHeight = window.innerHeight;
const currentScrollY = window.scrollY;

let targetY = currentScrollY + rect.top - (viewportHeight / 2) + (rect.height / 2);

// Add random deviation
const deviation = Math.random() * (viewportHeight * 0.05) + (viewportHeight * 0.05);
targetY += (Math.random() > 0.5 ? deviation : -deviation);

// Ensure bounds
if (targetY < 0) targetY = 0;
const maxScrollY = document.documentElement.scrollHeight - viewportHeight;
if (targetY > maxScrollY) targetY = maxScrollY;

return targetY;
EOF;

        return (int) $this->webDriver->executeScript($script, [$element]);
    }

    public function getMouseCoordinate(): Coordinate
    {
        return $this->mouseCoordinate;
    }

    public function getScrollCoordinate(): Coordinate
    {
        return $this->scrollCoordinate;
    }
}
