<?php

namespace Avodel\WebDriver\Components\Ajax;

use Avodel\WebDriver\Components\Ajax\Exception\AjaxResponseWasNotReceivedException;
use Behat\Mink\Session;
use Psr\Log\LoggerInterface;

final readonly class AjaxUtil
{
    public function __construct(
        private int $maxWaitingTimeMs,
        private LoggerInterface $logger,
    )
    {
    }

    public function overrideAjaxResponse(Session $session, string $method, string $url, string $body): void
    {
        $escapedBody = addslashes($body);
        $escapedBody = str_replace("\n", "\\n", $escapedBody);

        $statement = sprintf('overrideAjaxResponse(\'%s\', \'%s\', \'%s\');', $method, $url, $escapedBody);

        $session->executeScript($statement);
    }

    public function getLastAjaxResponse(Session $session, string $path): ?AjaxResponse
    {
        $responses = $session->evaluateScript('window.getAjaxResponses(\'' . $path . '\')');

        if (!$responses) {
            return null;
        }

        $lastResponse = $responses[array_key_last($responses)];

        return new AjaxResponse(
            (string)$lastResponse['url'],
            (int)$lastResponse['status'],
            (string)$lastResponse['content'],
            new \DateTimeImmutable($lastResponse['time']),
        );
    }

    public function getLastAjaxResponseMatchingPath(Session $session, string $path): ?AjaxResponse
    {
        $ajaxResponses = $this->getAllAjaxResponses($session);

        if (!$ajaxResponses) {
            return null;
        }

        $foundResponses = [];

        foreach ($ajaxResponses as $response) {
            $requestUrlData = parse_url($response->getUrl());
            $requestPath = $requestUrlData['path'] ?? '';
            if ($requestPath === $path) {
                $foundResponses[] = $response;
            }
        }

        return count($foundResponses) ? end($foundResponses) : null;
    }

    public function getAllRawAjaxResponses(Session $session): array
    {
$executable = <<<JS
return typeof window.getAllAjaxResponses === 'function' ? window.getAllAjaxResponses() : [];
JS;

        return $session->evaluateScript($executable);
    }

    /**
     * @return AjaxResponse[]
     */
    public function getAllAjaxResponses(Session $session): array
    {
        $endpointResponses = $this->getAllRawAjaxResponses($session);
        $result = [];

        foreach ($endpointResponses as $responses) {
            foreach ($responses as $response) {
                $result[] = new AjaxResponse(
                    (string)$response['url'],
                    (int)$response['status'],
                    (string)$response['content'],
                    new \DateTimeImmutable($response['time']),
                );
            }
        }

        return $result;
    }

    /**
     * @throws AjaxResponseWasNotReceivedException
     */
    public function waitForAjaxResponse(Session $session, string $path): void
    {
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The XHR request was not finished.', [
                    'path' => $path,
                    'executedTimeMs' => $executedTimeMs,
                ]);

                throw new AjaxResponseWasNotReceivedException(sprintf('The AJAX response was not received. Path: %s', $path));
            }

            $isLoadingActive = $session->evaluateScript(sprintf('window.isAjaxRequestActive("%s")', $path));

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    /**
     * @throws AjaxResponseWasNotReceivedException
     */
    public function waitUntilAllAjaxRequestsAreFinished(Session $session): void
    {
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The AJAX request was not finished.', [
                    'executedTimeMs' => $executedTimeMs,
                ]);

                throw new AjaxResponseWasNotReceivedException('The AJAX response was not received.');
            }

            $isLoadingActive = $session->evaluateScript('window.isAjaxRequestActive()');

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    public function isAjaxRequestActive(Session $session): bool
    {
        return $session->evaluateScript('window.isAjaxRequestActive()');
    }
}
