<?php

namespace Avodel\WebDriver\Components\Ajax;

use Avodel\WebDriver\Components\Ajax\Exception\AjaxResponseWasNotReceivedException;
use Avodel\WebDriver\Driver\MinkPhpWebDriver;
use Psr\Log\LoggerInterface;
use Symfony\Component\Serializer\SerializerInterface;

final readonly class AjaxHandler
{
    public function __construct(
        private MinkPhpWebDriver $driver,
        private int $maxWaitingTimeMs,
        private LoggerInterface $logger,
        private SerializerInterface $serializer,
    )
    {
    }

    private function register(): void
    {
        $registered = $this->driver->evaluateScript('window.ajaxUtilsRegistered');

        if ($registered === true) {
            return;
        }

        $this->driver->executeScript(file_get_contents(__DIR__ . '/../../Resources/js/ajaxoverrider.js'));
    }

    /**
     * Overrides an Ajax response with a custom response.
     */
    public function overrideAjaxResponse(string $method, string $url, string $body): void
    {
        $this->register();
        $escapedBody = addslashes($body);
        $escapedBody = str_replace("\n", "\\n", $escapedBody);

        $statement = sprintf('window.overrideAjaxResponse(\'%s\', \'%s\', \'%s\');', $method, $url, $escapedBody);

        $this->driver->executeScript($statement);
    }

    /**
     * Returns the last Ajax response for the given path.
     */
    public function getLastAjaxResponse(string $path): ?AjaxResponse
    {
        $this->register();
        $responses = $this->driver->evaluateScript('window.getAjaxResponses(\'' . $path . '\')');

        if (!$responses) {
            return null;
        }

        $lastResponse = $responses[array_key_last($responses)];

        return new AjaxResponse(
            (string)$lastResponse['url'],
            (int)$lastResponse['status'],
            (string)$lastResponse['content'],
            new \DateTimeImmutable($lastResponse['time']),
        );
    }

    /**
     * Returns the last Ajax response matching a specific base path excluding the query string.
     */
    public function getLastAjaxResponseMatchingPath(string $path): ?AjaxResponse
    {
        $this->register();
        $ajaxResponses = $this->getAllAjaxResponses();

        if (!$ajaxResponses) {
            return null;
        }

        $foundResponses = [];

        foreach ($ajaxResponses as $response) {
            $requestUrlData = parse_url($response->getUrl());
            $requestPath = $requestUrlData['path'] ?? '';
            if ($requestPath === $path) {
                $foundResponses[] = $response;
            }
        }

        return count($foundResponses) ? end($foundResponses) : null;
    }

    /**
     * @template T
     * @phpstan-template T
     *
     * @param class-string<T> $className
     *
     * @return T|null
     * @phpstan-return T|null
    */
    public function getLastDeserializedAjaxResponseMatchingPath(string $path, string $className, string $format = 'json'): mixed
    {
        $response = $this->getLastAjaxResponseMatchingPath($path);
        $content = $response?->getContent();

        if (!$content || trim($content) === '') {
            return null;
        }

        try {
            return $this->serializer->deserialize($content, $className, $format);
        } catch (\Exception $e) {
            $this->logger->error('Unable to deserialize ajax response.', [
                'path' => $path,
                'content' => substr($content, 0, 2000),
                'format' => $format,
                'class' => $className,
                'exception' => $e,
            ]);

            throw $e;
        }
    }

    /**
     * Returns all raw Ajax responses.
     */
    public function getAllRawAjaxResponses(): array
    {
        $this->register();
        $executable = <<<JS
return typeof window.getAllAjaxResponses === 'function' ? window.getAllAjaxResponses() : [];
JS;

        return $this->driver->evaluateScript($executable);
    }

    /**
     * Returns all Ajax responses.
     *
     * @return AjaxResponse[]
     */
    public function getAllAjaxResponses(): array
    {
        $endpointResponses = $this->getAllRawAjaxResponses();
        $result = [];

        foreach ($endpointResponses as $responses) {
            foreach ($responses as $response) {
                $result[] = new AjaxResponse(
                    (string)$response['url'],
                    (int)$response['status'],
                    (string)$response['content'],
                    new \DateTimeImmutable($response['time']),
                );
            }
        }

        return $result;
    }

    /**
     * Waits for an Ajax response to complete for the given path.
     *
     * @throws AjaxResponseWasNotReceivedException in case the AJAX response was not received
     */
    public function waitForAjaxResponse(string $path): void
    {
        $this->register();
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The XHR request was not finished.', [
                    'path' => $path,
                    'executedTimeMs' => $executedTimeMs,
                ]);

                throw new AjaxResponseWasNotReceivedException(sprintf('The AJAX response was not received. Path: %s', $path));
            }

            $isLoadingActive = $this->driver->evaluateScript(sprintf('window.isAjaxRequestActive("%s")', $path));

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    /**
     * Waits until all Ajax requests are finished.
     *
     * @throws AjaxResponseWasNotReceivedException in case the AJAX response was not received
     */
    public function waitUntilAllAjaxRequestsAreFinished(): void
    {
        $this->register();
        $t = microtime(true);

        while (true) {
            $executedTimeMs = (int)((microtime(true) - $t) * 1000);

            if ($executedTimeMs >= $this->maxWaitingTimeMs) {
                $this->logger->warning('The AJAX request was not finished.', [
                    'executedTimeMs' => $executedTimeMs,
                ]);

                throw new AjaxResponseWasNotReceivedException('The AJAX response was not received.');
            }

            $isLoadingActive = $this->driver->evaluateScript('window.isAjaxRequestActive()');

            if (!$isLoadingActive) {
                return;
            }

            usleep(200000);
        }
    }

    /**
     * Checks if an Ajax request is active.
     */
    public function isAjaxRequestActive(): bool
    {
        $this->register();
        return $this->driver->evaluateScript('window.isAjaxRequestActive()');
    }
}
