<?php

namespace Avodel\WebDriver\Keyboard;

use Behat\Mink\Exception\DriverException;
use Facebook\WebDriver\Chrome\ChromeDevToolsDriver;
use Facebook\WebDriver\Remote\RemoteWebDriver;
use Facebook\WebDriver\Remote\RemoteWebElement;

readonly class Keyboard
{
    public function __construct(
        /**
         * @@phpstan-ignore-next-line
         */
        private RemoteWebDriver $webDriver,
        /**
         * @@phpstan-ignore-next-line
         */
        private TypingAccuracyConfig $typingAccuracyConfig,
        /**
         * @@phpstan-ignore-next-line
         */
        private int $baseTypingCpm
    ) {
    }

//    /**
//     * @throws DriverException
//     */
    public function type(RemoteWebElement $element, string $text): void
    {
        $element->sendKeys($text);

//        $isFocused = $this->webDriver->executeScript(
//            'return arguments[0] === document.activeElement;',
//            [$element]
//        );
//
//        if (!$isFocused) {
//            throw new DriverException('Element is not focused when trying to type text.');
//        }
//
//        $baseDelay = (int)(60000 / $this->baseTypingCpm);
//        $dt = new ChromeDevToolsDriver($this->webDriver);
//
//        foreach (mb_str_split($text) as $char) {
//            $delay = $this->calculateTypingDelay($baseDelay);
//
//            $typoChance = 100 - (int)($this->typingAccuracyConfig->getAccuracy() * 100);
//
//            if (random_int(1, 100) <= $typoChance && (ctype_alpha($char) || ctype_digit($char))) {
//                $typoLength = random_int(1, 3);
//                $wrongChars = '';
//
//                for ($i = 0; $i < $typoLength; $i++) {
//                    if (ctype_digit($char)) {
//                        $wrongChars .= (string)random_int(0, 9);
//                    } else {
//                        $wrongChars .= chr(random_int(97, 122));
//                    }
//                }
//
//                foreach (mb_str_split($wrongChars) as $wrongChar) {
//                    $this->typeCharacterWithEvents($dt, $wrongChar);
//                    usleep($this->calculateTypingDelay($baseDelay));
//                }
//
//                usleep(random_int(
//                        $this->typingAccuracyConfig->getOnTypoDelay()->getMinMs(),
//                        $this->typingAccuracyConfig->getOnTypoDelay()->getMaxMs()
//                    ) * 1000);
//
//                for ($j = 0, $jMax = mb_strlen($wrongChars); $j < $jMax; $j++) {
//                    $this->pressBackspace($dt);
//                    usleep(random_int(
//                            $this->typingAccuracyConfig->getBackspaceDelay()->getMinMs(),
//                            $this->typingAccuracyConfig->getBackspaceDelay()->getMaxMs()
//                        ) * 1000);
//                }
//
//                usleep(random_int(
//                        $this->typingAccuracyConfig->getAfterTypoFixedDelay()->getMinMs(),
//                        $this->typingAccuracyConfig->getAfterTypoFixedDelay()->getMaxMs()
//                    ) * 1000);
//            }
//
//            $this->typeCharacterWithEvents($dt, $char);
//            usleep($delay);
//        }
    }

    /**
     * @@phpstan-ignore-next-line
     */
    private function typeCharacterWithEvents(ChromeDevToolsDriver $dt, string $char): void
    {
        $keyInfo = $this->getKeyInfo($char);
        $modifiers = $keyInfo['modifiers'];
        $shiftNeeded = $keyInfo['shift'];

        // Handle modifiers first
        if ($shiftNeeded) {
            $this->dispatchKeyEvent($dt, 'keyDown', 'Shift', 'ShiftLeft', 16, 8); // Shift modifier = 8
            usleep(random_int(1, 5) * 1000); // Small delay between modifier and key
        }

        // Raw key down (low-level event)
        $this->dispatchKeyEvent(
            $dt,
            'rawKeyDown',
            $keyInfo['key'],
            $keyInfo['code'],
            $keyInfo['keyCode'],
            $modifiers
        );

        // Key down (standard event)
        $this->dispatchKeyEvent(
            $dt,
            'keyDown',
            $keyInfo['key'],
            $keyInfo['code'],
            $keyInfo['keyCode'],
            $modifiers
        );

        // Character input (for printable characters)
        if (ord($char) >= 32 && ord($char) <= 126) {
            $dt->execute('Input.dispatchKeyEvent', [
                'type' => 'char',
                'text' => $char,
                'unmodifiedText' => $shiftNeeded ? strtolower($char) : $char,
                'modifiers' => $modifiers
            ]);
        }

        usleep(random_int(20, 60) * 1000); // Realistic key hold time

        // Key up
        $this->dispatchKeyEvent(
            $dt,
            'keyUp',
            $keyInfo['key'],
            $keyInfo['code'],
            $keyInfo['keyCode'],
            $modifiers
        );

        // Release modifiers
        if ($shiftNeeded) {
            usleep(random_int(1, 5) * 1000);
            $this->dispatchKeyEvent($dt, 'keyUp', 'Shift', 'ShiftLeft', 16, 0);
        }
    }

    private function dispatchKeyEvent(
        ChromeDevToolsDriver $dt,
        string $type,
        string $key,
        string $code,
        int $keyCode,
        int $modifiers
    ): void {
        $params = [
            'type' => $type,
            'key' => $key,
            'code' => $code,
            'windowsVirtualKeyCode' => $keyCode,
            'modifiers' => $modifiers
        ];

        // Add additional properties for keyDown and keyUp events
        if ($type === 'keyDown' || $type === 'keyUp') {
            $params['autoRepeat'] = false;
            $params['isKeypad'] = false;
            $params['isSystemKey'] = false;
        }

        // Add location for modifier keys
        if (in_array($key, ['Shift', 'Control', 'Alt', 'Meta'])) {
            $params['location'] = strpos($code, 'Left') !== false ? 1 : 2; // 1=left, 2=right
        }

        $dt->execute('Input.dispatchKeyEvent', $params);
    }

    /**
     * @@phpstan-ignore-next-line
     */
    private function pressBackspace(ChromeDevToolsDriver $dt): void
    {
        $this->dispatchKeyEvent($dt, 'keyDown', 'Backspace', 'Backspace', 8, 0);
        usleep(random_int(20, 50) * 1000);

        $this->dispatchKeyEvent($dt, 'keyUp', 'Backspace', 'Backspace', 8, 0);
    }

    private function getKeyInfo(string $char): array
    {
        $charCode = ord($char);
        $needsShift = false;
        $key = $char;
        $code = '';
        $keyCode = $charCode;

        // Handle special characters that need shift
        $shiftChars = [
            '!' => ['1', 'Digit1', 49],
            '@' => ['2', 'Digit2', 50],
            '#' => ['3', 'Digit3', 51],
            '$' => ['4', 'Digit4', 52],
            '%' => ['5', 'Digit5', 53],
            '^' => ['6', 'Digit6', 54],
            '&' => ['7', 'Digit7', 55],
            '*' => ['8', 'Digit8', 56],
            '(' => ['9', 'Digit9', 57],
            ')' => ['0', 'Digit0', 48],
            '_' => ['-', 'Minus', 45], // Fixed keyCode for underscore (Shift+-)
            '+' => ['=', 'Equal', 61], // Fixed keyCode for plus (Shift+=)
            '{' => ['[', 'BracketLeft', 91], // Fixed keyCode for { (Shift+[)
            '}' => [']', 'BracketRight', 93], // Fixed keyCode for } (Shift+])
            '|' => ['\\', 'Backslash', 92], // Fixed keyCode for | (Shift+\)
            ':' => [';', 'Semicolon', 59], // Fixed keyCode for : (Shift+;)
            '"' => ["'", 'Quote', 39], // Fixed keyCode for " (Shift+')
            '<' => [',', 'Comma', 44], // Fixed keyCode for < (Shift+,)
            '>' => ['.', 'Period', 46], // Fixed keyCode for > (Shift+.)
            '?' => ['/', 'Slash', 47], // Fixed keyCode for ? (Shift+/)
            '~' => ['`', 'Backquote', 96], // Fixed keyCode for ~ (Shift+`)
        ];

        if (isset($shiftChars[$char])) {
            $needsShift = true;
            $key = $shiftChars[$char][0];
            $code = $shiftChars[$char][1];
            $keyCode = $shiftChars[$char][2];
        } elseif (ctype_upper($char)) {
            $needsShift = true;
            $key = $char;
            $code = 'Key' . strtoupper($char);
            $keyCode = ord(strtoupper($char));
        } elseif (ctype_lower($char)) {
            $key = $char;
            $code = 'Key' . strtoupper($char);
            $keyCode = ord(strtoupper($char));
        } elseif (ctype_digit($char)) {
            $key = $char;
            $code = 'Digit' . $char;
            $keyCode = ord($char);
        } else {
            // Handle other special keys
            $specialKeys = [
                ' ' => ['Space', 'Space', 32],
                "\t" => ['Tab', 'Tab', 9],
                "\n" => ['Enter', 'Enter', 13],
                "\r" => ['Enter', 'Enter', 13],
                '-' => ['-', 'Minus', 45], // ASCII code for dash
                '=' => ['=', 'Equal', 61], // Fixed keyCode for equals
                '[' => ['[', 'BracketLeft', 91], // Fixed keyCode
                ']' => [']', 'BracketRight', 93], // Fixed keyCode
                '\\' => ['\\', 'Backslash', 92], // Fixed keyCode
                ';' => [';', 'Semicolon', 59], // Fixed keyCode
                "'" => ["'", 'Quote', 39], // Fixed keyCode
                ',' => [',', 'Comma', 44], // Fixed keyCode
                '.' => ['.', 'Period', 46], // Fixed keyCode
                '/' => ['/', 'Slash', 47], // Fixed keyCode
                '`' => ['`', 'Backquote', 96], // Fixed keyCode
            ];

            if (isset($specialKeys[$char])) {
                $key = $specialKeys[$char][0];
                $code = $specialKeys[$char][1];
                $keyCode = $specialKeys[$char][2];
            }
        }

        return [
            'key' => $key,
            'code' => $code,
            'keyCode' => $keyCode,
            'shift' => $needsShift,
            'modifiers' => $needsShift ? 8 : 0
        ];
    }

    /**
     * @@phpstan-ignore-next-line
     */
    private function calculateTypingDelay(int $baseDelay): int
    {
        $rand = random_int(0, 100);

        if ($rand < 60) {
            $delay = $baseDelay + random_int((int)(-$baseDelay * 0.3), (int)($baseDelay * 0.3));
        } elseif ($rand < 80) {
            $delay = $baseDelay + random_int($baseDelay, $baseDelay * 2);
        } else {
            $delayVal = max(5, (int)($baseDelay * 0.1));
            $delay = random_int(5, $delayVal);
        }

        return max(0, $delay) * 1000;
    }
}
